<?php
/**
 * Plugin Name: AegisShield Security
 * Plugin URI:  https://aegisify.com/wordpress-aegis-shield-security-product/
 * Description: AegisShield Security is a comprehensive WordPress security plugin providing activity logging, login protection, file integrity monitoring, security hardening, and proactive site defense tools to help safeguard your website.
 * Version:     7.1.16
 * Requires at least: 6.8
 * Requires PHP: 8.2
 * Author:      Aegisify, LLC
 * Author URI:  https://aegisify.com/
 * Text Domain: aegisshield-security
 * License: GPLv2 or later
 */

defined( 'ABSPATH' ) || exit;

define( 'AEGISSHIELD_VERSION', '7.1.16' );
define( 'AEGISSHIELD_PLUGIN_FILE', __FILE__ );
define( 'AEGISSHIELD_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AEGISSHIELD_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG' ) ) {
    define( 'AEGISSHIELD_UPDATE_DEBUG', ( defined( 'WP_DEBUG' ) && WP_DEBUG ) );
}

if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY' ) ) {
    define( 'AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY', 'aegisshield_update_check_last_error' );
}

if ( ! function_exists( 'aegisshield_internal_log' ) ) {
    function aegisshield_internal_log( $message, $context = array() ) {
        if ( ! defined( 'AEGISSHIELD_UPDATE_DEBUG' ) || ! AEGISSHIELD_UPDATE_DEBUG ) {
            return;
        }

        $prefix = '[AegisShield Update] ';
        if ( ! empty( $context ) && is_array( $context ) ) {
            $message .= ' | ' . wp_json_encode( $context );
        }
    }
}

if ( ! function_exists( 'aegisshield_update_set_error' ) ) {
    function aegisshield_update_set_error( $reason, $context = array() ) {
        set_transient(
            AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY,
            array(
                'time'    => gmdate( 'c' ),
                'reason'  => (string) $reason,
                'context' => is_array( $context ) ? $context : array(),
            ),
            15 * MINUTE_IN_SECONDS
        );
    }
}

require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-autoloader.php';
AegisShield\AS_Autoloader::init();

if ( ! function_exists( 'aegisshield_register_file_monitor_cron_schedules' ) ) {
	function aegisshield_register_file_monitor_cron_schedules( $schedules ) {

		if ( ! isset( $schedules['aegisshield_file_monitor_5min'] ) ) {
			$schedules['aegisshield_file_monitor_5min'] = array(
				'interval' => 5 * MINUTE_IN_SECONDS,
				'display'  => __( 'Every 5 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
			);
		}

		if ( ! isset( $schedules['aegisshield_file_monitor_10min'] ) ) {
			$schedules['aegisshield_file_monitor_10min'] = array(
				'interval' => 10 * MINUTE_IN_SECONDS,
				'display'  => __( 'Every 10 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
			);
		}

		if ( ! isset( $schedules['aegisshield_file_monitor_15min'] ) ) {
			$schedules['aegisshield_file_monitor_15min'] = array(
				'interval' => 15 * MINUTE_IN_SECONDS,
				'display'  => __( 'Every 15 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
			);
		}

		if ( ! isset( $schedules['aegisshield_file_monitor_30min'] ) ) {
			$schedules['aegisshield_file_monitor_30min'] = array(
				'interval' => 30 * MINUTE_IN_SECONDS,
				'display'  => __( 'Every 30 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
			);
		}

		if ( ! isset( $schedules['aegisshield_file_monitor_60min'] ) ) {
			$schedules['aegisshield_file_monitor_60min'] = array(
				'interval' => 60 * MINUTE_IN_SECONDS,
				'display'  => __( 'Every 60 minutes (AegisShield File Monitor)', 'aegisshield-security' ),
			);
		}

		return $schedules;
	}
}

add_filter( 'cron_schedules', 'aegisshield_register_file_monitor_cron_schedules' );
function aegisshield_security_bootstrap() {
    return AegisShield\AS_Plugin::instance();
}
add_action( 'init', 'aegisshield_security_bootstrap', 20 );
function aegisshield_security_activate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-plugin.php';
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';

    delete_transient( 'aegisshield_clm_free_reg_sent' );

    AegisShield\AS_Plugin::activate_static();
}
register_activation_hook( __FILE__, 'aegisshield_security_activate' );

function aegisshield_security_deactivate() {
    require_once AEGISSHIELD_PLUGIN_DIR . 'includes/class-as-cron.php';
    $cron = new AegisShield\AS_Cron();
    $cron->clear_events();
}
register_deactivation_hook( __FILE__, 'aegisshield_security_deactivate' );

if ( ! function_exists( 'aegisshield_is_pro_active' ) ) {
    function aegisshield_is_pro_active() {
        if ( ! class_exists( '\\AegisShield\\AS_Plugin' ) ) {
            return false;
        }

        $plugin = \AegisShield\AS_Plugin::instance();

        if ( method_exists( $plugin, 'get_license_manager' ) && $plugin->get_license_manager() ) {
            return (bool) apply_filters( 'aegisshield_is_pro_active', $plugin->get_license_manager()->is_pro_active() );
        }

        return false;
    }
}

if ( ! function_exists( 'aegisshield_show_pro_upgrade_notice' ) ) {
    add_action(
        'after_plugin_row_aegisshield-security/aegisshield-security.php',
        'aegisshield_show_pro_upgrade_notice',
        10,
        2
    );

    function aegisshield_show_pro_upgrade_notice( $plugin_file, $plugin_data ) {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        if ( function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active() ) {
            return;
        }

        echo '<tr class="plugin-update-tr"><td colspan="3" class="plugin-update colspanchange"><div class="update-message notice inline notice-warning notice-alt"><p>';

		$message = sprintf(
			__( '%1$sAegisShield Pro gives you our strongest security shield.%2$s Unlock advanced malware detection, automated deep scanning, database hardening automation, and extended activity logs to keep your site locked down on shared hosting. <a href="%3$s">Upgrade now to enable full protection.</a>', 'aegisshield-security' ),
			'<strong>',
			'</strong>',
			esc_url( admin_url( 'admin.php?page=aegisshield-license' ) )
		);

        echo wp_kses_post( $message );
        echo '</p></div></td></tr>';
    }
}

add_action( 'admin_notices', function () {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $last = get_transient( AEGISSHIELD_UPDATE_DEBUG_TRANSIENT_KEY );
    if ( empty( $last ) || ! is_array( $last ) ) {
        return;
    }

    $time   = ! empty( $last['time'] ) ? $last['time'] : '';
    $reason = ! empty( $last['reason'] ) ? $last['reason'] : 'Unknown';

    echo '<div class="notice notice-error"><p>';
    echo '<strong>AegisShield update check failed.</strong> ';
    echo esc_html( $reason );
    if ( $time ) {
        echo ' (UTC: ' . esc_html( $time ) . ')';
    }
    echo ' — check <code>wp-content/debug.log</code> for details.';
    echo '</p></div>';
} );

function aegisshield_clm_register_free_install() {

		if ( get_option( 'aegisshield_clm_free_reg_done' ) === '1' ) {
			update_option( 'aegisshield_clm_free_reg_last_error', '', false );
			return true;
		}

		if ( get_option( 'aegisshield_clm_free_reg_consent' ) !== '1' ) {
			update_option( 'aegisshield_clm_free_reg_last_error', 'Consent not provided.', false );
			return false;
		}

		$next_allowed = (int) get_option( 'aegisshield_clm_free_reg_next_allowed', 0 );
		if ( $next_allowed > 0 && time() < $next_allowed ) {
			update_option( 'aegisshield_clm_free_reg_last_error', 'Registration temporarily throttled. Please try again later.', false );
			return false;
		}

		$lock_ts = (int) get_option( 'aegisshield_clm_free_reg_lock_ts', 0 );
		if ( $lock_ts > 0 && ( time() - $lock_ts ) < 60 ) {
			return;
		}
		update_option( 'aegisshield_clm_free_reg_lock_ts', time(), false );

		if ( get_transient( 'aegisshield_clm_free_reg_sent' ) ) {
			update_option( 'aegisshield_clm_free_reg_last_error', 'Registration recently attempted. Please wait a few minutes.', false );
			update_option( 'aegisshield_clm_free_reg_lock_ts', 0, false );
			return false;
		}

		$endpoint = 'https://aegisify.com/wp-json/chue-license/v1/aegisshield/free-register';

    $site_url = home_url();
    $domain   = wp_parse_url( $site_url, PHP_URL_HOST );

    $payload = array(
        'product'        => 'aegisshield',
        'email'          => get_option( 'admin_email' ),
        'site_url'       => $site_url,
        'domain'         => $domain ? $domain : '',
        'plugin_version' => defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '',
        'wp_version'     => get_bloginfo( 'version' ),
        'php_version'    => PHP_VERSION,
    );

    $args = array(
        'timeout' => 20,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => wp_json_encode( $payload ),
    );

	$resp = wp_remote_post( $endpoint, $args );

	if ( is_wp_error( $resp ) ) {

		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		$msg = 'CLM free-register failed: ' . $resp->get_error_message();
		update_option( 'aegisshield_clm_free_reg_last_error', $msg, false );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// Debug log removed for production (Plugin Check).
		}
		return false;
	}

	$code = (int) wp_remote_retrieve_response_code( $resp );
	$body = (string) wp_remote_retrieve_body( $resp );

	if ( $code < 200 || $code >= 300 ) {
		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		$msg = 'CLM free-register HTTP ' . $code;
		update_option( 'aegisshield_clm_free_reg_last_error', $msg, false );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// Debug log removed for production (Plugin Check).
		}
		return false;
	}

	$parsed = json_decode( $body, true );
	if ( ! is_array( $parsed ) ) {
		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );

		$msg = 'CLM free-register returned invalid JSON.';
		update_option( 'aegisshield_clm_free_reg_last_error', $msg, false );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		// Debug log removed for production (Plugin Check).
		}
		return false;
	}


	if ( ! empty( $parsed['success'] ) ) {

		update_option( 'aegisshield_clm_free_reg_done', '1', false );

		update_option( 'aegisshield_clm_free_reg_next_allowed', time() + DAY_IN_SECONDS, false );

		set_transient( 'aegisshield_clm_free_reg_sent', 1, DAY_IN_SECONDS );

	} else {

		if ( $code === 429 ) {
			update_option( 'aegisshield_clm_free_reg_next_allowed', time() + ( 60 * 60 ), false ); // 1 hour
		} else {
			update_option( 'aegisshield_clm_free_reg_next_allowed', time() + ( 15 * 60 ), false ); // 15 minutes
		}

		set_transient( 'aegisshield_clm_free_reg_sent', 1, 15 * MINUTE_IN_SECONDS );
	}

	update_option( 'aegisshield_clm_free_reg_lock_ts', 0, false );

	if ( get_option( 'aegisshield_clm_free_reg_done' ) === '1' ) {
		update_option( 'aegisshield_clm_free_reg_last_error', '', false );
		return true;
	}

	update_option( 'aegisshield_clm_free_reg_last_error', 'CLM did not confirm registration.', false );
	return false;
}

function aegisshield_clm_unregister_free_install() {

	delete_option( 'aegisshield_clm_free_reg_done' );
	delete_option( 'aegisshield_clm_free_reg_next_allowed' );
	delete_option( 'aegisshield_clm_free_reg_lock_ts' );
	delete_option( 'aegisshield_clm_free_reg_last_error' );

	delete_transient( 'aegisshield_clm_free_reg_sent' );

	return true;
}

add_filter( 'update_plugins_transient_hook', 'aegisshield_updates_disabled' );