<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_DB_Tools;

defined( 'ABSPATH' ) || exit;

class AS_Page_Dashboard {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }
		// ---------------------------------------------------------------------
		// Dashboard Charts (Threat & High-Activity Intelligence)
		// ---------------------------------------------------------------------
			if ( ! wp_script_is( 'aegisshield-chartjs', 'enqueued' ) ) {

				// Local-first (WordPress.org compliant: do not load external CDN assets).
				// This file lives in /includes/admin_pages/, so we need 3 levels up to reach plugin root.
				$plugin_root = dirname( __FILE__, 3 );
				$chartjs_rel = 'assets/js/chart.umd.min.js';
				$chartjs_abs = $plugin_root . '/' . $chartjs_rel;

				if ( file_exists( $chartjs_abs ) ) {
					$chartjs_url = plugins_url( $chartjs_rel, $plugin_root . '/aegisshield-security.php' );

					wp_enqueue_script(
						'aegisshield-chartjs',
						$chartjs_url,
						array(),
						'4.4.1',
						true
					);
				} else {
					// Local Chart.js asset missing - charts will remain disabled (no remote scripts).
					// Intentionally do not enqueue a CDN resource.
				}
			}

		if ( ! wp_script_is( 'aegisshield-dashboard-charts', 'enqueued' ) ) {
			wp_register_script(
				'aegisshield-dashboard-charts',
				'',
				array( 'aegisshield-chartjs' ),
				'1.0.0',
				true
			);
			wp_enqueue_script( 'aegisshield-dashboard-charts' );
		}

        $db_widget      = $this->build_db_health_widget();
        $health_metrics = $this->build_core_health_metrics();

        $license_manager = ( $this->plugin && method_exists( $this->plugin, 'get_license_manager' ) )
            ? $this->plugin->get_license_manager()
            : null;

        $is_pro = ( $license_manager && method_exists( $license_manager, 'is_pro_active' ) )
            ? (bool) $license_manager->is_pro_active()
            : false;

        // ---------------------------------------------------------------------
        // Security score calculation (high‑level approximate risk indicator).
        // ---------------------------------------------------------------------
        $score        = 100;
        $score_reason = array();

        if ( isset( $db_widget['prefix_status'] ) && false !== strpos( $db_widget['prefix_status'], 'default' ) ) {
            $score -= 15;
            $score_reason[] = __( 'Database is still using the default table prefix.', 'aegisshield-security' );
        }

        if ( isset( $db_widget['spike_count'] ) && (int) $db_widget['spike_count'] > 0 ) {
            $score -= 10;
            $score_reason[] = __( 'One or more tables show significant growth since the last snapshot.', 'aegisshield-security' );
        }

        if ( ! empty( $health_metrics['plugin_updates'] ) ) {
            $score -= min( 20, (int) $health_metrics['plugin_updates'] * 2 );
            $score_reason[] = sprintf(
                /* translators: %d: number of plugins with updates. */
                _n( '%d plugin has a pending update.', '%d plugins have pending updates.', (int) $health_metrics['plugin_updates'], 'aegisshield-security' ),
                (int) $health_metrics['plugin_updates']
            );
        }

        if ( ! empty( $health_metrics['theme_updates'] ) ) {
            $score -= min( 10, (int) $health_metrics['theme_updates'] * 3 );
            $score_reason[] = sprintf(
                /* translators: %d: number of themes with updates. */
                _n( '%d theme has a pending update.', '%d themes have pending updates.', (int) $health_metrics['theme_updates'], 'aegisshield-security' ),
                (int) $health_metrics['theme_updates']
            );
        }

        if ( isset( $health_metrics['admin_users'] ) && $health_metrics['admin_users'] > 3 ) {
            $score -= 10;
            $score_reason[] = __( 'There are more than three administrator accounts.', 'aegisshield-security' );
        }

        if ( isset( $health_metrics['php_version'] ) && version_compare( $health_metrics['php_version'], '8.0', '<' ) ) {
            $score -= 10;
            $score_reason[] = __( 'The site is running an older PHP version. Consider upgrading.', 'aegisshield-security' );
        }

        if ( $score < 0 ) {
            $score = 0;
        } elseif ( $score > 100 ) {
            $score = 100;
        }

        if ( $score >= 90 ) {
            $score_label = __( 'Secure', 'aegisshield-security' );
            $score_class = 'aegisshield-score-good';
        } elseif ( $score >= 70 ) {
            $score_label = __( 'Needs attention', 'aegisshield-security' );
            $score_class = 'aegisshield-score-warn';
        } else {
            $score_label = __( 'High risk', 'aegisshield-security' );
            $score_class = 'aegisshield-score-bad';
        }

        // ---------------------------------------------------------------------
        // Phase 2: Deep scan summaries (File Integrity + Malware / Quick Scan).
        // ---------------------------------------------------------------------
        $settings = ( $this->plugin && method_exists( $this->plugin, 'get_settings' ) )
            ? $this->plugin->get_settings()
            : null;

        $fi_summary_line   = '';
        $mw_summary_line   = '';
        $attack_story_line = __( 'Attack Story correlates malware hits, file changes, and admin activity into a single timeline so you can see how an intrusion unfolded.', 'aegisshield-security' );

        if ( $settings ) {
            // File Integrity.
            $fi_report = $settings->get( 'file_integrity', 'last_scan_report', array() );
            if ( ! is_array( $fi_report ) ) {
                $fi_report = array();
            }

            $fi_mode   = isset( $fi_report['mode'] ) ? (string) $fi_report['mode'] : '';
            $fi_stats  = ( isset( $fi_report['stats'] ) && is_array( $fi_report['stats'] ) ) ? $fi_report['stats'] : array();
            $fi_high   = isset( $fi_report['high'] ) ? (int) $fi_report['high'] : 0;
            $fi_medium = isset( $fi_report['medium'] ) ? (int) $fi_report['medium'] : 0;
            $fi_low    = isset( $fi_report['low'] ) ? (int) $fi_report['low'] : 0;
            $fi_time   = isset( $fi_report['timestamp'] ) ? (string) $fi_report['timestamp'] : '';

            $fi_scanned  = isset( $fi_stats['scanned'] ) ? (int) $fi_stats['scanned'] : 0;
            $fi_modified = isset( $fi_stats['modified'] ) ? (int) $fi_stats['modified'] : 0;
            $fi_new      = isset( $fi_stats['new'] ) ? (int) $fi_stats['new'] : 0;

            $fi_mode_label = '';
            if ( 'light' === $fi_mode ) {
                $fi_mode_label = __( 'Light', 'aegisshield-security' );
            } elseif ( 'hybrid' === $fi_mode ) {
                $fi_mode_label = __( 'Hybrid', 'aegisshield-security' );
            } elseif ( 'full' === $fi_mode ) {
                $fi_mode_label = __( 'Full', 'aegisshield-security' );
            }

            $fi_time_label = '';
            if ( ! empty( $fi_time ) ) {
                $timestamp = strtotime( $fi_time );
                if ( $timestamp ) {
                    $fi_time_label = date_i18n(
                        get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                        $timestamp
                    );
                } else {
                    $fi_time_label = $fi_time;
                }
            }

            if ( ! empty( $fi_report ) && $fi_scanned > 0 ) {
                $fi_summary_line = sprintf(
                    /* translators: 1: scan mode, 2: scanned, 3: modified, 4: new, 5: high, 6: medium, 7: low, 8: time. */
                    __( 'File Integrity: %1$s scan, %2$d files scanned, %3$d modified, %4$d new. Findings: %5$d high / %6$d medium / %7$d low. Last run: %8$s.', 'aegisshield-security' ),
                    $fi_mode_label ? $fi_mode_label : __( 'Unknown mode', 'aegisshield-security' ),
                    $fi_scanned,
                    $fi_modified,
                    $fi_new,
                    $fi_high,
                    $fi_medium,
                    $fi_low,
                    $fi_time_label ? $fi_time_label : __( 'n/a', 'aegisshield-security' )
                );
            } else {
                $fi_summary_line = __( 'File Integrity: no scans have been recorded yet.', 'aegisshield-security' );
            }

            // Malware / Quick Scan.
            $malware_meta = $settings->get( 'malware', 'last_results_meta', array() );
            if ( ! is_array( $malware_meta ) ) {
                $malware_meta = array();
            }

            $mw_type       = isset( $malware_meta['scan_type'] ) ? (string) $malware_meta['scan_type'] : '';
            $mw_started    = isset( $malware_meta['started_at'] ) ? (int) $malware_meta['started_at'] : 0;
            $mw_completed  = isset( $malware_meta['completed_at'] ) ? (int) $malware_meta['completed_at'] : 0;
            $mw_files      = isset( $malware_meta['file_count'] ) ? (int) $malware_meta['file_count'] : 0;
            $mw_suspects   = isset( $malware_meta['suspect_count'] ) ? (int) $malware_meta['suspect_count'] : 0;
            $mw_is_partial = ! empty( $malware_meta['partial'] );

            $mw_label_type = '';
            if ( false !== strpos( $mw_type, 'quick' ) ) {
                $mw_label_type = __( 'Quick malware scan (changed files only)', 'aegisshield-security' );
            } elseif ( 'manual' === $mw_type ) {
                $mw_label_type = __( 'Manual full malware scan', 'aegisshield-security' );
            } elseif ( 'auto' === $mw_type ) {
                $mw_label_type = __( 'Automatic full malware scan', 'aegisshield-security' );
            } elseif ( ! empty( $mw_type ) ) {
                $mw_label_type = $mw_type;
            }

            $mw_time_label = '';
            if ( $mw_completed ) {
                $mw_time_label = date_i18n(
                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                    $mw_completed
                );
            }

            if ( ! empty( $malware_meta ) && $mw_files > 0 ) {
                $mw_summary_line = sprintf(
                    /* translators: 1: scan label, 2: files, 3: suspects, 4: completed time, 5: partial note. */
                    __( 'Malware / Quick Scan: %1$s scanned %2$d files, %3$d suspects. Completed: %4$s%5$s', 'aegisshield-security' ),
                    $mw_label_type ? $mw_label_type : __( 'Latest scan', 'aegisshield-security' ),
                    $mw_files,
                    $mw_suspects,
                    $mw_time_label ? $mw_time_label : __( 'n/a', 'aegisshield-security' ),
                    $mw_is_partial ? ' ' . __( '(partial quick scan – remaining files will be scanned later)', 'aegisshield-security' ) : ''
                );
            } else {
                $mw_summary_line = __( 'Malware / Quick Scan: no recent scan metadata found.', 'aegisshield-security' );
            }
        } else {
            $fi_summary_line = __( 'File Integrity: settings are not available.', 'aegisshield-security' );
            $mw_summary_line = __( 'Malware / Quick Scan: settings are not available.', 'aegisshield-security' );
        }

        // ---------------------------------------------------------------------
        // Phase 3: Activity stats + Alert Center preview.
        // ---------------------------------------------------------------------
        $activity_stats = $this->get_recent_event_stats();
        $recent_alerts  = $this->get_recent_alert_events( 10 );

        ?>
        <div class="wrap aegisshield-dashboard-wrap">
            <h1><?php esc_html_e( 'AegisShield Security Dashboard', 'aegisshield-security' ); ?></h1>

            <p class="aegisshield-dashboard-intro">
                <?php esc_html_e( 'The AegisShield Dashboard provides a centralized, high-level view of your WordPress site’s security posture by combining system health, active protections, recent events, and detected risks into a single, easy-to-understand interface. It helps administrators quickly assess overall security status, identify issues that need attention, and confidently navigate deeper security tools when action is required—without needing to sift through complex logs or technical data.', 'aegisshield-security' ); ?>
            </p>

			<?php

			global $wpdb;

			$table = $this->sanitize_table_name( $this->get_activity_log_table_name() );

			$since_24h = gmdate( 'Y-m-d H:i:s', time() - 24 * 3600 );
			$since_7d  = gmdate( 'Y-m-d H:i:s', time() - 7 * 24 * 3600 );
			$since_14d = gmdate( 'Y-m-d H:i:s', time() - 14 * 24 * 3600 );

			$prevent_params = array(
				'login_blocked_lockout',
				'login_blocked_%',
				'%blocked%',
				'%prevent%',
				'%denied%',
				'%violation%',
				'%throttle%',
				'%rate_limit%',
				'%ratelimit%',
				'%challenge%',
				'%captcha%',
				'%ban%',
				'%drop%',
				'waf_%',
				'bot_%',
				'ddos_%',
			);

			// NOTE: The prevented/blocked filter is embedded as a fixed SQL clause in each prepared statement
			// to avoid dynamic WHERE string building and to keep PluginCheck/PHPCS satisfied.

$blocked_24h = 0;
			if ( ! empty( $table ) ) {
				$args_blocked_24h = array_merge( array( $table, $since_24h ), $prevent_params );
				$blocked_24h = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT COUNT(*) FROM %i
						 WHERE event_time >= %s
						 AND ( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s )",
						$table,
						$since_24h,
						$prevent_params[0] ?? '',
						$prevent_params[1] ?? '',
						$prevent_params[2] ?? '',
						$prevent_params[3] ?? '',
						$prevent_params[4] ?? '',
						$prevent_params[5] ?? '',
						$prevent_params[6] ?? '',
						$prevent_params[7] ?? '',
						$prevent_params[8] ?? '',
						$prevent_params[9] ?? '',
						$prevent_params[10] ?? '',
						$prevent_params[11] ?? '',
						$prevent_params[12] ?? '',
						$prevent_params[13] ?? '',
						$prevent_params[14] ?? '',
						$prevent_params[15] ?? ''
					)
				);
			}

$threat_since = ( $blocked_24h < 10 ) ? $since_7d : $since_24h;

// If very low volume, group by day; otherwise group by hour.
$blocked_group_expr = ( $blocked_24h < 20 )
	? "DATE(event_time)"
	: "DATE_FORMAT(event_time, '%%Y-%%m-%%d %%H:00:00')";

			// Initialize payload
			$payload = array(
				'blocked_area' => array( 'labels' => array(), 'counts' => array() ),        // Chart 1
				'top_ips'      => array( 'labels' => array(), 'counts' => array() ),        // Chart 2
				'threat_donut' => array( 'labels' => array(), 'counts' => array() ),        // Chart 3
				'auth_line'    => array( 'labels' => array(), 'failed' => array(), 'lockouts' => array(), 'success' => array() ), // Chart 4
				'sys_stack'    => array( 'labels' => array(), 'plugin_on' => array(), 'plugin_off' => array(), 'admin_add' => array(), 'admin_del' => array() ), // Chart 5
				'risk_radar'   => array( 'labels' => array( 'Login Guard','File Integrity','Malware','Security Headers','Hardening','DB Tools','Activity Log' ), 'counts' => array( 0,0,0,0,0,0,0 ) ), // Chart 6
				'fim_scatter'  => array(),                                                  // Chart 7
				'db_pie'       => array( 'labels' => array(), 'counts' => array() ),        // Chart 8
			);

			if ( ! empty( $table ) ) {

				// -------------------------------------------------------------
				// (1) AREA: Prevented/Blocked actions by hour (last 24h)
				// -------------------------------------------------------------
					$rows_blocked_args = array_merge( array( $table, $threat_since ), $prevent_params );
				if ( $blocked_24h < 20 ) {
					$rows_blocked = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
						$wpdb->prepare(
							"SELECT DATE(event_time) AS h, COUNT(*) AS c
							 FROM %i
							 WHERE event_time >= %s
							 AND ( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s )
							 GROUP BY h
							 ORDER BY h ASC",
							$table,
							$threat_since,
							$prevent_params[0] ?? '',
							$prevent_params[1] ?? '',
							$prevent_params[2] ?? '',
							$prevent_params[3] ?? '',
							$prevent_params[4] ?? '',
							$prevent_params[5] ?? '',
							$prevent_params[6] ?? '',
							$prevent_params[7] ?? '',
							$prevent_params[8] ?? '',
							$prevent_params[9] ?? '',
							$prevent_params[10] ?? '',
							$prevent_params[11] ?? '',
							$prevent_params[12] ?? '',
							$prevent_params[13] ?? '',
							$prevent_params[14] ?? '',
							$prevent_params[15] ?? ''
						),
						ARRAY_A
					);
				} else {
					$rows_blocked = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
						$wpdb->prepare(
							"SELECT DATE_FORMAT(event_time, '%%Y-%%m-%%d %%H:00:00') AS h, COUNT(*) AS c
							 FROM %i
							 WHERE event_time >= %s
							 AND ( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s )
							 GROUP BY h
							 ORDER BY h ASC",
							$table,
							$threat_since,
							$prevent_params[0] ?? '',
							$prevent_params[1] ?? '',
							$prevent_params[2] ?? '',
							$prevent_params[3] ?? '',
							$prevent_params[4] ?? '',
							$prevent_params[5] ?? '',
							$prevent_params[6] ?? '',
							$prevent_params[7] ?? '',
							$prevent_params[8] ?? '',
							$prevent_params[9] ?? '',
							$prevent_params[10] ?? '',
							$prevent_params[11] ?? '',
							$prevent_params[12] ?? '',
							$prevent_params[13] ?? '',
							$prevent_params[14] ?? '',
							$prevent_params[15] ?? ''
						),
						ARRAY_A
					);
				}

				foreach ( (array) $rows_blocked as $r ) {
					$payload['blocked_area']['labels'][] = $r['h'];
					$payload['blocked_area']['counts'][] = (int) $r['c'];
				}

				// -------------------------------------------------------------
				// (2) BAR: Top hostile IPs (last 24h, prevented only)
				// -------------------------------------------------------------
				$rows_ips = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT ip_address AS ip, COUNT(*) AS c
						 FROM %i
						 WHERE event_time >= %s
						 AND ip_address <> ''
						 AND ( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s )
						 GROUP BY ip_address
						 ORDER BY c DESC
						 LIMIT 10",
						$table,
						$threat_since,
						$prevent_params[0] ?? '',
						$prevent_params[1] ?? '',
						$prevent_params[2] ?? '',
						$prevent_params[3] ?? '',
						$prevent_params[4] ?? '',
						$prevent_params[5] ?? '',
						$prevent_params[6] ?? '',
						$prevent_params[7] ?? '',
						$prevent_params[8] ?? '',
						$prevent_params[9] ?? '',
						$prevent_params[10] ?? '',
						$prevent_params[11] ?? '',
						$prevent_params[12] ?? '',
						$prevent_params[13] ?? '',
						$prevent_params[14] ?? '',
						$prevent_params[15] ?? ''
					),
					ARRAY_A
				);

				foreach ( (array) $rows_ips as $r ) {
					$payload['top_ips']['labels'][] = (string) $r['ip'];
					$payload['top_ips']['counts'][] = (int) $r['c'];
				}

				// -------------------------------------------------------------
				// (3) DONUT: Threat categories by module (last 24h, high signal)
				// -------------------------------------------------------------
				$like_login         = 'login_%';
				$like_mfa           = 'mfa_%';
				$like_login_blocked = 'login_blocked_%';
				$like_fim           = 'fim_%';
				$like_integrity     = 'integrity_%';
				$like_file_integrity = 'file_integrity_%';
				$like_malware       = 'malware_%';
				$like_headers       = 'headers_%';
				$like_csp           = 'csp_%';
				$like_hardening     = 'hardening_%';
				$like_db_tools      = 'db_tools_%';
				$like_db            = 'db_%';

					$args_threats = array_merge( array( $like_login, $like_mfa, $like_login_blocked, $like_fim, $like_integrity, $like_file_integrity, $like_malware, $like_headers, $like_csp, $like_hardening, $like_db_tools, $like_db, $table, $since_24h ), $prevent_params, array( 'login_failed', $like_malware, $like_fim, $like_hardening, $like_headers, $like_csp, $like_db_tools, $like_db ) );
				$rows_threats = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT
							CASE
								WHEN event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s THEN 'Login Guard'
								WHEN event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type = 'file_integrity_changed' THEN 'File Integrity'
								WHEN event_type LIKE %s THEN 'Malware Scan'
								WHEN event_type LIKE %s OR event_type LIKE %s THEN 'Security Headers'
								WHEN event_type LIKE %s THEN 'Hardening'
								WHEN event_type LIKE %s OR event_type LIKE %s THEN 'DB Tools'
								ELSE 'Other'
							END AS bucket,
							COUNT(*) AS c
						 FROM %i
						 WHERE event_time >= %s
						 AND (
							( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s )
							OR event_type = %s
							OR event_type LIKE %s
							OR event_type LIKE %s
							OR event_type LIKE %s
							OR event_type LIKE %s
							OR event_type LIKE %s
							OR event_type LIKE %s
							OR event_type LIKE %s
						 )
						 GROUP BY bucket
						 ORDER BY c DESC",
						$like_login,
						$like_mfa,
						$like_login_blocked,
						$like_fim,
						$like_integrity,
						$like_file_integrity,
						$like_malware,
						$like_headers,
						$like_csp,
						$like_hardening,
						$like_db_tools,
						$like_db,
						$table,
						$since_24h,
						$prevent_params[0] ?? '',
						$prevent_params[1] ?? '',
						$prevent_params[2] ?? '',
						$prevent_params[3] ?? '',
						$prevent_params[4] ?? '',
						$prevent_params[5] ?? '',
						$prevent_params[6] ?? '',
						$prevent_params[7] ?? '',
						$prevent_params[8] ?? '',
						$prevent_params[9] ?? '',
						$prevent_params[10] ?? '',
						$prevent_params[11] ?? '',
						$prevent_params[12] ?? '',
						$prevent_params[13] ?? '',
						$prevent_params[14] ?? '',
						$prevent_params[15] ?? '',
						'login_failed',
						$like_malware,
						$like_fim,
						$like_hardening,
						$like_headers,
						$like_csp,
						$like_db_tools,
						$like_db
					),
					ARRAY_A
				);

				foreach ( (array) $rows_threats as $r ) {
					$payload['threat_donut']['labels'][] = (string) $r['bucket'];
					$payload['threat_donut']['counts'][] = (int) $r['c'];
				}

				// -------------------------------------------------------------
				// (4) LINE: Auth pressure hourly (success/failed/lockouts) 24h
				// -------------------------------------------------------------
				$rows_auth = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT DATE_FORMAT(event_time, '%%Y-%%m-%%d %%H:00:00') AS h,
								SUM(CASE WHEN event_type='login_failed'  THEN 1 ELSE 0 END) AS failed,
								SUM(CASE WHEN event_type='login_lockout' THEN 1 ELSE 0 END) AS lockouts,
								SUM(CASE WHEN event_type='login_success' THEN 1 ELSE 0 END) AS success
						 FROM %i
						 WHERE event_time >= %s
						 GROUP BY h
						 ORDER BY h ASC",
						$table, $since_24h
					),
					ARRAY_A
				);

				foreach ( (array) $rows_auth as $r ) {
					$payload['auth_line']['labels'][]   = $r['h'];
					$payload['auth_line']['failed'][]   = (int) $r['failed'];
					$payload['auth_line']['lockouts'][] = (int) $r['lockouts'];
					$payload['auth_line']['success'][]  = (int) $r['success'];
				}

				// -------------------------------------------------------------
				// (5) STACKED BAR: High-impact system changes (last 7d by day)
				// -------------------------------------------------------------
				$rows_sys = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT DATE(event_time) AS d,
								SUM(CASE WHEN event_type='plugin_activated'   THEN 1 ELSE 0 END) AS plugin_on,
								SUM(CASE WHEN event_type='plugin_deactivated' THEN 1 ELSE 0 END) AS plugin_off,
								SUM(CASE WHEN event_type='admin_created'      THEN 1 ELSE 0 END) AS admin_add,
								SUM(CASE WHEN event_type='admin_deleted'      THEN 1 ELSE 0 END) AS admin_del
						 FROM %i
						 WHERE event_time >= %s
						 GROUP BY d
						 ORDER BY d ASC",
						$table, $since_7d
					),
					ARRAY_A
				);

				foreach ( (array) $rows_sys as $r ) {
					$payload['sys_stack']['labels'][]     = $r['d'];
					$payload['sys_stack']['plugin_on'][]  = (int) $r['plugin_on'];
					$payload['sys_stack']['plugin_off'][] = (int) $r['plugin_off'];
					$payload['sys_stack']['admin_add'][]  = (int) $r['admin_add'];
					$payload['sys_stack']['admin_del'][]  = (int) $r['admin_del'];
				}

				// -------------------------------------------------------------
				// (6) RADAR: Module risk index (last 7d)
				// Risk = prevented/blocked/failed/violation-like activity per module.
				// -------------------------------------------------------------
				$like_login_guard = 'login_%';
				$like_mfa_guard   = 'mfa_%';
				$like_fim2        = 'fim_%';
				$like_integrity2  = 'integrity_%';
				$like_malware2    = 'malware_%';
				$like_headers2    = 'headers_%';
				$like_csp2        = 'csp_%';
				$like_hardening2  = 'hardening_%';
				$like_db_tools2   = 'db_tools_%';
				$like_db2         = 'db_%';

					$args_risk = array_merge( array( $like_login_guard, $like_mfa_guard ), $prevent_params, array( 'login_failed', 'login_lockout', $like_fim2, $like_integrity2, $like_malware2, $like_headers2, $like_csp2, $like_hardening2, $like_db_tools2, $like_db2, $table, $since_7d ) );
				$rows_risk = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT
							SUM(CASE WHEN (event_type LIKE %s OR event_type LIKE %s) AND ( ( event_type = %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s OR event_type LIKE %s ) OR event_type = %s OR event_type = %s ) THEN 1 ELSE 0 END) AS login_guard,
							SUM(CASE WHEN (event_type LIKE %s OR event_type LIKE %s) THEN 1 ELSE 0 END) AS fim,
							SUM(CASE WHEN (event_type LIKE %s) THEN 1 ELSE 0 END) AS malware,
							SUM(CASE WHEN (event_type LIKE %s OR event_type LIKE %s) THEN 1 ELSE 0 END) AS headers,
							SUM(CASE WHEN (event_type LIKE %s) THEN 1 ELSE 0 END) AS hardening,
							SUM(CASE WHEN (event_type LIKE %s OR event_type LIKE %s) THEN 1 ELSE 0 END) AS dbtools,
							SUM(CASE WHEN (event_type IN ('plugin_activated','plugin_deactivated','admin_created','admin_deleted')) THEN 1 ELSE 0 END) AS activity
						 FROM %i
						 WHERE event_time >= %s",
						$like_login_guard,
						$like_mfa_guard,
						$prevent_params[0] ?? '',
						$prevent_params[1] ?? '',
						$prevent_params[2] ?? '',
						$prevent_params[3] ?? '',
						$prevent_params[4] ?? '',
						$prevent_params[5] ?? '',
						$prevent_params[6] ?? '',
						$prevent_params[7] ?? '',
						$prevent_params[8] ?? '',
						$prevent_params[9] ?? '',
						$prevent_params[10] ?? '',
						$prevent_params[11] ?? '',
						$prevent_params[12] ?? '',
						$prevent_params[13] ?? '',
						$prevent_params[14] ?? '',
						$prevent_params[15] ?? '',
						'login_failed',
						'login_lockout',
						$like_fim2,
						$like_integrity2,
						$like_malware2,
						$like_headers2,
						$like_csp2,
						$like_hardening2,
						$like_db_tools2,
						$like_db2,
						$table,
						$since_7d
					),
					ARRAY_A
				);

				if ( ! empty( $rows_risk[0] ) ) {
					$payload['risk_radar']['counts'] = array(
						(int) $rows_risk[0]['login_guard'],
						(int) $rows_risk[0]['fim'],
						(int) $rows_risk[0]['malware'],
						(int) $rows_risk[0]['headers'],
						(int) $rows_risk[0]['hardening'],
						(int) $rows_risk[0]['dbtools'],
						(int) $rows_risk[0]['activity'],
					);
				}

				// -------------------------------------------------------------
				// (7) SCATTER: File Integrity bursts (14d) — points per day
				// -------------------------------------------------------------
				$rows_fim = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT DATE(event_time) AS d, COUNT(*) AS c
						 FROM %i
						 WHERE event_time >= %s
						 AND (event_type LIKE %s OR event_type LIKE %s)
						 GROUP BY d
						 ORDER BY d ASC",
						$table, $since_14d, 'fim_%', 'integrity_%'
					),
					ARRAY_A
				);

				foreach ( (array) $rows_fim as $r ) {
					$payload['fim_scatter'][] = array(
						'x' => $r['d'],
						'y' => (int) $r['c'],
					);
				}

				// -------------------------------------------------------------
				// (8) PIE: DB Tools high-activity actions (last 7d, top 8)
				// -------------------------------------------------------------
				$rows_db = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
					$wpdb->prepare(
						"SELECT event_type AS t, COUNT(*) AS c
						 FROM %i
						 WHERE event_time >= %s
						 AND (event_type LIKE %s OR event_type LIKE %s)
						 GROUP BY event_type
						 ORDER BY c DESC
						 LIMIT 8",
						$table, $since_7d, $like_db_tools2, $like_db2
					),
					ARRAY_A
				);

				foreach ( (array) $rows_db as $r ) {
					$payload['db_pie']['labels'][] = (string) $r['t'];
					$payload['db_pie']['counts'][] = (int) $r['c'];
				}
			}

			$charts_json = wp_json_encode( $payload );

			$js = "
			(function(){
			  function ready(fn){ if(document.readyState!=='loading'){fn();} else {document.addEventListener('DOMContentLoaded', fn);} }
			  ready(function(){
				if(typeof Chart==='undefined'){ return; }
				var d = {$charts_json};

				Chart.defaults.responsive = true;
				Chart.defaults.maintainAspectRatio = false;

				// 1) AREA (blocked/prevented)
				var a = document.getElementById('as_dash_threat_area');
				if(a){
				  new Chart(a,{
					type:'line',
					data:{ labels:d.blocked_area.labels, datasets:[{ label:'Prevented', data:d.blocked_area.counts, tension:0.25, fill:true }]},
					options:{ plugins:{ legend:{ display:false } }, scales:{ y:{ beginAtZero:true } } }
				  });
				}

				// 2) BAR (top hostile IPs)
				var b = document.getElementById('as_dash_threat_ips');
				if(b){
				  new Chart(b,{
					type:'bar',
					data:{ labels:d.top_ips.labels, datasets:[{ label:'Blocked', data:d.top_ips.counts }]},
					options:{ plugins:{ legend:{ display:false } }, scales:{ y:{ beginAtZero:true } } }
				  });
				}

				// 3) DONUT (threat categories)
				var c = document.getElementById('as_dash_threat_donut');
				if(c){
				  new Chart(c,{
					type:'doughnut',
					data:{ labels:d.threat_donut.labels, datasets:[{ data:d.threat_donut.counts }]},
					options:{ plugins:{ legend:{ position:'bottom' } } }
				  });
				}

				// 4) LINE (auth pressure)
				var l = document.getElementById('as_dash_auth_line');
				if(l){
				  new Chart(l,{
					type:'line',
					data:{
					  labels:d.auth_line.labels,
					  datasets:[
						{ label:'Failed',   data:d.auth_line.failed,   tension:0.25, fill:false },
						{ label:'Lockouts', data:d.auth_line.lockouts, tension:0.25, fill:false },
						{ label:'Success',  data:d.auth_line.success,  tension:0.25, fill:false }
					  ]
					},
					options:{ plugins:{ legend:{ position:'bottom' } }, scales:{ y:{ beginAtZero:true } } }
				  });
				}

				// 5) STACKED BAR (system changes)
				var s = document.getElementById('as_dash_sys_stack');
				if(s){
				  new Chart(s,{
					type:'bar',
					data:{
					  labels:d.sys_stack.labels,
					  datasets:[
						{ label:'Plugin Enabled',  data:d.sys_stack.plugin_on,  stack:'sys' },
						{ label:'Plugin Disabled', data:d.sys_stack.plugin_off, stack:'sys' },
						{ label:'Admin Added',     data:d.sys_stack.admin_add,  stack:'sys' },
						{ label:'Admin Removed',   data:d.sys_stack.admin_del,  stack:'sys' }
					  ]
					},
					options:{ plugins:{ legend:{ position:'bottom' } }, scales:{ x:{ stacked:true }, y:{ stacked:true, beginAtZero:true } } }
				  });
				}

				// 6) RADAR (module risk)
				var r = document.getElementById('as_dash_risk_radar');
				if(r){
				  new Chart(r,{
					type:'radar',
					data:{ labels:d.risk_radar.labels, datasets:[{ label:'Risk', data:d.risk_radar.counts, fill:true }]},
					options:{ plugins:{ legend:{ display:false } } }
				  });
				}

				// 7) SCATTER (FIM bursts)
				var f = document.getElementById('as_dash_fim_scatter');
				if(f){
				  new Chart(f,{
					type:'scatter',
					data:{ datasets:[{ label:'File Integrity Events', data:d.fim_scatter }]},
					options:{
					  plugins:{ legend:{ display:false } },
					  scales:{
						x:{ type:'category', title:{ display:true, text:'Date' } },
						y:{ beginAtZero:true }
					  }
					}
				  });
				}

				// 8) PIE (DB tools actions)
				var p = document.getElementById('as_dash_db_pie');
				if(p){
				  new Chart(p,{
					type:'pie',
					data:{ labels:d.db_pie.labels, datasets:[{ data:d.db_pie.counts }]},
					options:{ plugins:{ legend:{ position:'bottom' } } }
				  });
				}
			  });
			})();";

			wp_add_inline_script( 'aegisshield-dashboard-charts', $js );
			?>

			<style>
			  .as-dash-threat-wrap{ margin:10px 0 16px; }
			  .as-dash-threat-title{
				text-align:center; font-weight:700; font-size:16px; margin:0 0 10px; color:#d63638;
			  }
			  .as-dash-threat-grid{
				display:grid;
				grid-template-columns: repeat(4, minmax(0, 1fr));
				gap:12px;
				align-items:stretch;
			  }
			  .as-dash-threat-card{
				background:#fff; border:1px solid #dcdcde; border-radius:10px; padding:10px;
			  }
			  .as-dash-threat-card h3{ margin:0 0 8px; font-size:13px; font-weight:600; }
			  .as-dash-threat-canvas{ height:175px; position:relative; }

			  @media (max-width: 1400px){
				.as-dash-threat-grid{ grid-template-columns: repeat(2, minmax(0, 1fr)); }
			  }
			  @media (max-width: 860px){
				.as-dash-threat-grid{ grid-template-columns: 1fr; }
			  }
			</style>

			<div class="as-dash-threat-wrap">
			  <div class="as-dash-threat-title"><?php esc_html_e( 'Intelligent Threat & High Activity Charts', 'aegisshield-security' ); ?></div>

			  <div class="as-dash-threat-grid">
				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'Prevented / Blocked Actions (24h)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_threat_area"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'Top Hostile IPs (24h)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_threat_ips"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'Threat Categories by Module (24h)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_threat_donut"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'Auth Pressure (24h)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_auth_line"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'High-Impact Changes (7d)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_sys_stack"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'Module Risk Index (7d)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_risk_radar"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'File Integrity Bursts (14d)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_fim_scatter"></canvas></div>
				</div>

				<div class="as-dash-threat-card">
				  <h3><?php esc_html_e( 'DB Tools High Activity (7d)', 'aegisshield-security' ); ?></h3>
				  <div class="as-dash-threat-canvas"><canvas id="as_dash_db_pie"></canvas></div>
				</div>
			  </div>
			</div>

            <style>
                .aegisshield-dashboard-wrap .aegisshield-section {
                    margin-top: 20px;
                }
                .aegisshield-dashboard-wrap .aegisshield-flex-row {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel {
                    background: #ffffff;
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
                    padding: 16px 18px;
                    box-sizing: border-box;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-half {
                    flex: 1 1 48%;
                    min-width: 260px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-full {
                    flex: 1 1 100%;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-header {
                    display: flex;
                    align-items: center;
                    margin-bottom: 8px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-title {
                    font-size: 15px;
                    font-weight: 600;
                    margin: 0;
                    color: #1d2327;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-title-icon {
                    margin-right: 6px;
                }
                .aegisshield-dashboard-wrap .aegisshield-panel-subtitle {
                    font-size: 12px;
                    color: #555d66;
                    margin: 0 0 8px 0;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-value {
                    font-size: 26px;
                    font-weight: 700;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-label {
                    font-size: 13px;
                    margin-top: 4px;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-good .aegisshield-score-value {
                    color: #008a20;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-warn .aegisshield-score-value {
                    color: #d98300;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-bad .aegisshield-score-value {
                    color: #d63638;
                }
                .aegisshield-dashboard-wrap .aegisshield-score-reasons {
                    margin: 8px 0 0 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-list {
                    margin: 0;
                    padding: 0;
                    list-style: none;
                    font-size: 12px;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-list li {
                    display: flex;
                    justify-content: space-between;
                    padding: 2px 0;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-label {
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-kv-value {
                    font-weight: 600;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-panel {
                    border-left: 4px solid #2271b1;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-header-badge {
                    display: inline-block;
                    margin-left: 8px;
                    padding: 2px 8px;
                    border-radius: 20px;
                    font-size: 10px;
                    font-weight: 600;
                    text-transform: uppercase;
                    background: #2271b1;
                    color: #ffffff;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-grid {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                    margin-top: 10px;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel {
                    flex: 1 1 32%;
                    min-width: 230px;
                    background: #f8fafc;
                    border-radius: 4px;
                    padding: 10px 12px;
                    box-sizing: border-box;
                    border: 1px solid #e2e8f0;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-title {
                    font-size: 13px;
                    font-weight: 600;
                    margin: 0 0 4px 0;
                    color: #1d2327;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-subpanel-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-dimmed {
                    opacity: 0.6;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-footer {
                    margin-top: 10px;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    font-size: 12px;
                }
                .aegisshield-dashboard-wrap .aegisshield-pro-footer .button-primary {
                    background: #2271b1;
                    border-color: #2271b1;
                }
                .aegisshield-dashboard-wrap .aegisshield-events-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-alert-list {
                    margin: 0;
                    padding-left: 18px;
                    font-size: 12px;
                    color: #555d66;
                }
                .aegisshield-dashboard-wrap .aegisshield-alert-list li span {
                    font-size: 11px;
                    color: #6c757d;
                }
                .aegisshield-dashboard-wrap .aegisshield-modules-grid {
                    display: flex;
                    flex-wrap: wrap;
                    gap: 16px;
                    margin-top: 10px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-card {
                    flex: 1 1 32%;
                    min-width: 230px;
                    background: #ffffff;
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
                    padding: 12px 14px;
                    box-sizing: border-box;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-header {
                    display: flex;
                    align-items: center;
                    margin-bottom: 4px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-icon {
                    margin-right: 6px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-title {
                    font-weight: 600;
                    margin: 0;
                    font-size: 13px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-status {
                    margin-left: auto;
                    font-size: 11px;
                    padding: 2px 6px;
                    border-radius: 3px;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-ok {
                    background: #e7f7ec;
                    color: #008a20;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-warn {
                    background: #fff4e5;
                    color: #d98300;
                }
                .aegisshield-dashboard-wrap .aegisshield-status-bad {
                    background: #f5c2c7;
                    color: #842029;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-body {
                    font-size: 12px;
                    color: #555d66;
                    margin-bottom: 8px;
                }
                .aegisshield-dashboard-wrap .aegisshield-module-actions .button.button-small {
                    font-size: 11px;
                    padding: 2px 8px;
                    height: auto;
                    line-height: 18px;
                }
                @media (max-width: 960px) {
                    .aegisshield-dashboard-wrap .aegisshield-panel-half {
                        flex: 1 1 100%;
                    }
                    .aegisshield-dashboard-wrap .aegisshield-pro-subpanel {
                        flex: 1 1 100%;
                    }
                    .aegisshield-dashboard-wrap .aegisshield-module-card {
                        flex: 1 1 100%;
                    }
                }
            </style>

            <div class="aegisshield-section">
                <div class="aegisshield-flex-row">
                    <div class="aegisshield-panel aegisshield-panel-half <?php echo esc_attr( $score_class ); ?>">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🛡️</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Overall Security Score', 'aegisshield-security' ); ?></h2>
                        </div>
                        <div class="aegisshield-panel-body">
                            <div class="aegisshield-score-value">
                                <?php
                                printf(
                                    /* translators: %d: security score. */
                                    esc_html__( '%d/100', 'aegisshield-security' ),
                                    (int) $score
                                );
                                ?>
                            </div>
                            <div class="aegisshield-score-label">
                                <?php echo esc_html( $score_label ); ?>
                            </div>
                            <?php if ( ! empty( $score_reason ) ) : ?>
                                <ul class="aegisshield-score-reasons">
                                    <?php foreach ( $score_reason as $reason ) : ?>
                                        <li><?php echo esc_html( $reason ); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php else : ?>
                                <p class="aegisshield-panel-subtitle">
                                    <?php esc_html_e( 'All high‑level checks look good. Keep monitoring modules below for changes.', 'aegisshield-security' ); ?>
                                </p>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🧩</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Core Health Metrics', 'aegisshield-security' ); ?></h2>
                        </div>
                        <p class="aegisshield-panel-subtitle">
                            <?php esc_html_e( 'These metrics summarize your WordPress, PHP, and database state at a glance.', 'aegisshield-security' ); ?>
                        </p>
                        <ul class="aegisshield-kv-list">
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'WordPress version', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( $health_metrics['wp_version'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'PHP version', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( $health_metrics['php_version'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Plugins with updates', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['plugin_updates'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Themes with updates', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['theme_updates'] ); ?>
                                </span>
                            </li>
                            <li>
                                <span class="aegisshield-kv-label"><?php esc_html_e( 'Administrator accounts', 'aegisshield-security' ); ?></span>
                                <span class="aegisshield-kv-value">
                                    <?php echo esc_html( (int) $health_metrics['admin_users'] ); ?>
                                </span>
                            </li>
                            <?php if ( ! empty( $db_widget['total_size_human'] ) ) : ?>
                                <li>
                                    <span class="aegisshield-kv-label"><?php esc_html_e( 'Database size', 'aegisshield-security' ); ?></span>
                                    <span class="aegisshield-kv-value"><?php echo esc_html( $db_widget['total_size_human'] ); ?></span>
                                </li>
                            <?php endif; ?>
                            <?php if ( ! empty( $db_widget['prefix_status'] ) ) : ?>
                                <li>
                                    <span class="aegisshield-kv-label"><?php esc_html_e( 'DB prefix', 'aegisshield-security' ); ?></span>
                                    <span class="aegisshield-kv-value"><?php echo esc_html( $db_widget['prefix_status'] ); ?></span>
                                </li>
                            <?php endif; ?>
                        </ul>
                    </div>
                </div>
            </div>

            <div class="aegisshield-section">
                <div class="aegisshield-panel aegisshield-panel-full aegisshield-pro-panel<?php echo esc_attr( $is_pro ? '' : ' aegisshield-pro-dimmed' ); ?>">
                    <div class="aegisshield-panel-header">
                        <span class="aegisshield-panel-title-icon">🤖</span>
                        <h2 class="aegisshield-panel-title">
                            <?php esc_html_e( 'AegisShield Pro Intelligence', 'aegisshield-security' ); ?>
                            <span class="aegisshield-pro-header-badge"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
                        </h2>
                    </div>
                    <p class="aegisshield-panel-subtitle">
                        <?php esc_html_e( 'Aggregated intelligence from Activity Log, Login Guard, File Integrity, Malware scanning, and Security Headers into one premium overview.', 'aegisshield-security' ); ?>
                    </p>

                    <div class="aegisshield-pro-subpanel-grid">
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Enhanced system health metrics', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li>
                                    <?php
                                    printf(
                                        /* translators: 1: admin user count. */
                                        esc_html__( 'Admin accounts: %1$d total (keep this as low as practical).', 'aegisshield-security' ),
                                        (int) $health_metrics['admin_users']
                                    );
                                    ?>
                                </li>
                                <li>
                                    <?php
                                    printf(
                                        /* translators: 1: plugin updates, 2: theme updates. */
                                        esc_html__( 'Updates: %1$d plugins and %2$d themes have pending updates.', 'aegisshield-security' ),
                                        (int) $health_metrics['plugin_updates'],
                                        (int) $health_metrics['theme_updates']
                                    );
                                    ?>
                                </li>
                                <?php if ( ! empty( $db_widget['largest_table'] ) ) : ?>
                                    <li>
                                        <?php
                                        printf(
                                            /* translators: 1: largest table name, 2: largest table size. */
                                            esc_html__( 'Largest DB table: %1$s (%2$s).', 'aegisshield-security' ),
                                            esc_html( $db_widget['largest_table'] ),
                                            ! empty( $db_widget['largest_table_size_human'] ) ? esc_html( $db_widget['largest_table_size_human'] ) : esc_html__( 'size unknown', 'aegisshield-security' )
                                        );
                                        ?>
                                    </li>
                                <?php endif; ?>
                                <li>
                                    <?php
                                    printf(
                        /* translators: %d: number of tables with growth spikes. */
                                        esc_html__( 'Database growth spikes detected: %d table(s).', 'aegisshield-security' ),
                                        isset( $db_widget['spike_count'] ) ? (int) $db_widget['spike_count'] : 0
                                    );
                                    ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Hardening and Security Headers modules help reduce attack surface and enforce browser‑side protections.', 'aegisshield-security' ); ?>
                                </li>
                            </ul>
                        </div>

                        <!-- Deep scan summaries -->
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Deep scan summaries (Integrity, Quick Scan, Attack Story)', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li><?php echo esc_html( $fi_summary_line ); ?></li>
                                <li><?php echo esc_html( $mw_summary_line ); ?></li>
                                <li><?php echo esc_html( $attack_story_line ); ?></li>
                            </ul>
                        </div>

                        <!-- Pro module indicators -->
                        <div class="aegisshield-pro-subpanel">
                            <p class="aegisshield-pro-subpanel-title">
                                <?php esc_html_e( 'Pro module indicators (CSP Builder, Attack Story, etc.)', 'aegisshield-security' ); ?>
                            </p>
                            <ul class="aegisshield-pro-subpanel-list">
                                <li>
                                    <?php esc_html_e( 'CSP Builder Pro & header profiles for advanced Security Headers management.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Attack Story view on Malware page to reconstruct multi‑step attacks (file changes + malware hits).', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Incremental Quick Scan that focuses on changed files using File Monitor baselines.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Alert rules on Activity Log to send notifications when critical events occur.', 'aegisshield-security' ); ?>
                                </li>
                                <li>
                                    <?php esc_html_e( 'Email alert logic for file changes and high‑risk events.', 'aegisshield-security' ); ?>
                                </li>
                            </ul>
                        </div>
                    </div>

                    <div class="aegisshield-pro-footer">
                        <?php if ( $is_pro ) : ?>
                            <span><?php esc_html_e( 'Your AegisShield Pro license is active. All Pro intelligence modules are available.', 'aegisshield-security' ); ?></span>
                        <?php else : ?>
                            <span><?php esc_html_e( 'Activate AegisShield Pro to unlock deep scan summaries, Attack Story, and advanced header / alert controls.', 'aegisshield-security' ); ?></span>
                            <a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>">
                                <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Section 3: Security Events + Alert Center -->
            <div class="aegisshield-section">
                <div class="aegisshield-flex-row">
                    <!-- Security events (last 24 hours) -->
                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">📈</span>
                            <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Security events (last 24 hours)', 'aegisshield-security' ); ?></h2>
                        </div>
                        <?php
                        $failed        = isset( $activity_stats['login_failed'] ) ? (int) $activity_stats['login_failed'] : 0;
                        $success       = isset( $activity_stats['login_success'] ) ? (int) $activity_stats['login_success'] : 0;
                        $lockouts      = isset( $activity_stats['login_lockout'] ) ? (int) $activity_stats['login_lockout'] : 0;
                        $admin_created = isset( $activity_stats['admin_created'] ) ? (int) $activity_stats['admin_created'] : 0;
                        $admin_deleted = isset( $activity_stats['admin_deleted'] ) ? (int) $activity_stats['admin_deleted'] : 0;
                        $admin_changes = $admin_created + $admin_deleted;
                        ?>
                        <ul class="aegisshield-events-list">
                            <li><?php esc_html_e( 'Failed logins:', 'aegisshield-security' ); ?> <?php echo esc_html( $failed ); ?></li>
                            <li><?php esc_html_e( 'Login lockouts:', 'aegisshield-security' ); ?> <?php echo esc_html( $lockouts ); ?></li>
                            <li><?php esc_html_e( 'Successful logins:', 'aegisshield-security' ); ?> <?php echo esc_html( $success ); ?></li>
                            <li><?php esc_html_e( 'Admin account changes:', 'aegisshield-security' ); ?> <?php echo esc_html( $admin_changes ); ?></li>
                        </ul>
                    </div>

                    <!-- Alert Center preview -->
                    <div class="aegisshield-panel aegisshield-panel-half">
                        <div class="aegisshield-panel-header">
                            <span class="aegisshield-panel-title-icon">🚨</span>
                            <h2 class="aegisshield-panel-title">
                                <?php esc_html_e( 'Alert Center preview', 'aegisshield-security' ); ?>
                            </h2>
                        </div>

                        <?php if ( $is_pro ) : ?>
                            <?php if ( ! empty( $recent_alerts ) ) : ?>
                                <ul class="aegisshield-alert-list">
                                    <?php foreach ( $recent_alerts as $event ) : ?>
                                        <?php
                                        $user_label = '';
                                        if ( ! empty( $event->user_id ) ) {
                                            $user = get_userdata( (int) $event->user_id );
                                            if ( $user && ! is_wp_error( $user ) ) {
                                                $user_label = $user->user_login;
                                            }
                                        }

                                        $time_label = $event->event_time;
                                        if ( ! empty( $event->event_time ) ) {
                                            $ts = strtotime( $event->event_time );
                                            if ( $ts ) {
                                                $time_label = date_i18n(
                                                    get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
                                                    $ts
                                                );
                                            }
                                        }
                                        ?>
                                        <li>
                                            <strong><?php echo esc_html( $event->event_type ); ?></strong>
                                            &mdash;
                                            <?php echo esc_html( wp_trim_words( $event->message, 12, '…' ) ); ?>
                                            <?php if ( $user_label ) : ?>
                                                (<?php echo esc_html( $user_label ); ?>)
                                            <?php endif; ?>
                                            <?php if ( ! empty( $event->ip_address ) ) : ?>
                                                &middot; <?php echo esc_html( $event->ip_address ); ?>
                                            <?php endif; ?>
                                            &middot; <span><?php echo esc_html( $time_label ); ?></span>
                                        </li>
                                    <?php endforeach; ?>
                                </ul>
                            <?php else : ?>
                                <p class="aegisshield-panel-subtitle">
                                    <?php esc_html_e( 'No recent alert‑level events have been logged yet.', 'aegisshield-security' ); ?>
                                </p>
                            <?php endif; ?>
                        <?php else : ?>
                            <p class="aegisshield-panel-subtitle">
                                <?php esc_html_e( 'Alert Center is part of AegisShield Pro. It surfaces high‑impact security events such as malware detections, lockouts, and admin changes in one feed.', 'aegisshield-security' ); ?>
                            </p>
                            <a class="button button-primary" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>">
                                <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Section 4: Modules overview -->
            <div class="aegisshield-section">
                <div class="aegisshield-panel aegisshield-panel-full">
                    <div class="aegisshield-panel-header">
                        <span class="aegisshield-panel-title-icon">🧭</span>
                        <h2 class="aegisshield-panel-title"><?php esc_html_e( 'Modules overview', 'aegisshield-security' ); ?></h2>
                    </div>
                    <p class="aegisshield-panel-subtitle">
                        <?php esc_html_e( 'Jump directly into key security modules below.', 'aegisshield-security' ); ?>
                    </p>

                    <div class="aegisshield-modules-grid">
                        <!-- Activity Log -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-list-view aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Activity Log', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Track important security‑related events such as logins, changes, and administrative actions.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-activity-log' ) ); ?>">
                                    <?php esc_html_e( 'View Activity Log', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Login Guard -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-shield-alt aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Login Guard', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Protects your login form against brute‑force attempts by tracking failures and blocking abusive IPs.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-login-guard' ) ); ?>">
                                    <?php esc_html_e( 'View Login Guard', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- File Integrity -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-media-code aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'File Integrity', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Monitors key files for unexpected changes, additions, or removals that may indicate tampering.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-file-integrity' ) ); ?>">
                                    <?php esc_html_e( 'Open File Integrity', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Hardening -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-lock aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Hardening', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Applies recommended WordPress hardening such as disabling file editors and tightening entry points.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-hardening' ) ); ?>">
                                    <?php esc_html_e( 'Open Hardening', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Security Headers -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-admin-network aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Security Headers', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Helps send modern browser security headers such as X‑Frame‑Options, Referrer‑Policy, and HSTS.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-sec-headers' ) ); ?>">
                                    <?php esc_html_e( 'Open Security Headers', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Malware -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-warning aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Malware tools', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status aegisshield-status-ok">
                                    <?php esc_html_e( 'OK', 'aegisshield-security' ); ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Scans for suspicious code patterns and known malware signatures.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-malware' ) ); ?>">
                                    <?php esc_html_e( 'Open Malware Tools', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>

                        <!-- Database Tools -->
                        <div class="aegisshield-module-card">
                            <div class="aegisshield-module-header">
                                <span class="dashicons dashicons-database aegisshield-module-icon" aria-hidden="true"></span>
                                <h3 class="aegisshield-module-title"><?php esc_html_e( 'Database Tools', 'aegisshield-security' ); ?></h3>
                                <span class="aegisshield-module-status <?php echo ( ! empty( $db_widget['spike_count'] ) ) ? 'aegisshield-status-warn' : 'aegisshield-status-ok'; ?>">
                                    <?php
                                    if ( ! empty( $db_widget['spike_count'] ) ) {
                                        esc_html_e( 'Check growth', 'aegisshield-security' );
                                    } else {
                                        esc_html_e( 'OK', 'aegisshield-security' );
                                    }
                                    ?>
                                </span>
                            </div>
                            <div class="aegisshield-module-body">
                                <?php esc_html_e( 'Review table sizes, growth, and perform cleanup or maintenance operations.', 'aegisshield-security' ); ?>
                            </div>
                            <div class="aegisshield-module-actions">
                                <a class="button button-small" href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-db-tools' ) ); ?>">
                                    <?php esc_html_e( 'Open DB Tools', 'aegisshield-security' ); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    protected function build_core_health_metrics() {
        $wp_version  = get_bloginfo( 'version' );
        $php_version = PHP_VERSION;

        // Plugin update count.
        $plugin_updates      = 0;
        $plugins_update_data = get_site_transient( 'update_plugins' );
        if ( is_object( $plugins_update_data ) && ! empty( $plugins_update_data->response ) && is_array( $plugins_update_data->response ) ) {
            $plugin_updates = count( $plugins_update_data->response );
        }

        // Theme update count.
        $theme_updates     = 0;
        $theme_update_data = get_site_transient( 'update_themes' );
        if ( is_object( $theme_update_data ) && ! empty( $theme_update_data->response ) && is_array( $theme_update_data->response ) ) {
            $theme_updates = count( $theme_update_data->response );
        }

        // Administrator accounts.
        $admin_users = 0;
        if ( function_exists( 'get_users' ) ) {
            $admins = get_users(
                array(
                    'role'   => 'administrator',
                    'fields' => 'ID',
                )
            );
            if ( is_array( $admins ) ) {
                $admin_users = count( $admins );
            }
        }

        return array(
            'wp_version'     => $wp_version,
            'php_version'    => $php_version,
            'plugin_updates' => (int) $plugin_updates,
            'theme_updates'  => (int) $theme_updates,
            'admin_users'    => (int) $admin_users,
        );
    }

    /**
     * Sanitize a table name used in direct SQL strings.
     *
     * Table names cannot be passed as placeholders in $wpdb->prepare(), so we must strictly validate
     * the identifier to prevent SQL injection. We only allow alphanumeric + underscore and require
     * the table to be within the current WordPress table prefix (including multisite base prefix).
     *
     * @param string $table Raw table name.
     * @return string Sanitized table name, or empty string if invalid.
     */
    
    /**
     * Sanitize a database prefix once (identifier-safe).
     *
     * @param string $prefix Raw prefix.
     * @return string Sanitized prefix.
     */
    protected function sanitize_db_prefix( $prefix ) {
        $prefix = (string) $prefix;
        // Allow only identifier-safe chars.
        $prefix = preg_replace( '/[^A-Za-z0-9_]/', '', $prefix );
        return (string) $prefix;
    }

    /**
     * Escape an SQL identifier using a strict allowlist and backtick wrapping.
     *
     * @param string $identifier Identifier (table/column).
     * @return string Escaped identifier or empty string if invalid.
     */
    protected function escape_sql_identifier( $identifier ) {
        $identifier = (string) $identifier;
        if ( '' === $identifier ) {
            return '';
        }
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $identifier ) ) {
            return '';
        }
        return '`' . $identifier . '`';
    }

    /**
     * Replace a {{table}} token in SQL with a safely escaped identifier.
     *
     * @param string $sql SQL containing {{table}}.
     * @param string $table Table name identifier (unescaped).
     * @return string SQL with escaped table identifier.
     */
    protected function sql_with_table( $sql, $table ) {
        $escaped = $this->escape_sql_identifier( $table );
        if ( '' === $escaped ) {
            return (string) $sql;
        }
        return str_replace( '{{table}}', $escaped, (string) $sql );
    }

protected function sanitize_table_name( $table ) {
        global $wpdb;

        $table = is_string( $table ) ? trim( $table ) : '';
        if ( '' === $table ) {
            return '';
        }

        // Remove common identifier wrappers / unsafe characters.
        $table = str_replace( array( '`', ' ', "\t", "\n", "\r", ';' ), '', $table );

        // Only allow [A-Za-z0-9_].
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $table ) ) {
            return '';
        }

        $prefix      = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';
        $base_prefix = isset( $wpdb->base_prefix ) ? (string) $wpdb->base_prefix : '';

        // Require the table to live inside the current WP table namespace.
        if ( '' !== $prefix && 0 === strpos( $table, $prefix ) ) {
            return $table;
        }

        if ( '' !== $base_prefix && 0 === strpos( $table, $base_prefix ) ) {
            return $table;
        }

        return '';
    }

    protected function get_activity_log_table_name() {
        global $wpdb;

        if ( ! $this->plugin || ! method_exists( $this->plugin, 'get_logger' ) ) {
            return '';
        }

        $logger = $this->plugin->get_logger();
        if ( $logger && method_exists( $logger, 'get_table_name' ) ) {
            return $this->sanitize_table_name( $logger->get_table_name() );
        }

        return $wpdb->prefix . 'aegisshield_activity_log';
    }

    protected function get_recent_event_stats( $seconds = 0 ) {
        global $wpdb;

        $table = $this->sanitize_table_name( $this->get_activity_log_table_name() );
        if ( empty( $table ) ) {
            return array();
        }

        $default_window = defined( 'DAY_IN_SECONDS' ) ? DAY_IN_SECONDS : 86400;
        $seconds        = absint( $seconds );
        if ( $seconds <= 0 ) {
            $seconds = $default_window;
        }

        $since = gmdate( 'Y-m-d H:i:s', time() - $seconds );

        $types = array(
            'login_failed',
            'login_success',
            'login_lockout',
            'plugin_activated',
            'plugin_deactivated',
            'admin_created',
            'admin_deleted',
        );

        $args_recent_stats = array_merge( array( $table, $since ), $types );

        $cache_key_recent_stats = 'aegisshield_dash_recent_stats_' . md5( wp_json_encode( array( $table, $since, $types ) ) );


        $rows = wp_cache_get( $cache_key_recent_stats, 'aegisshield' );


        if ( false === $rows ) {

		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct DB access is required for custom tables.
			$wpdb->prepare(
				"SELECT event_type, COUNT(*) AS cnt
				 FROM {$table}
				 WHERE event_time >= %s
				   AND event_type IN (%s,%s,%s,%s,%s,%s,%s)
				 GROUP BY event_type",
				$since,
				$types[0] ?? '',
				$types[1] ?? '',
				$types[2] ?? '',
				$types[3] ?? '',
				$types[4] ?? '',
				$types[5] ?? '',
				$types[6] ?? ''
			),
			OBJECT
		);

        	wp_cache_set( $cache_key_recent_stats, $rows, 'aegisshield', MINUTE_IN_SECONDS );


        }
        $stats = array_fill_keys( $types, 0 );

        if ( $rows ) {
            foreach ( $rows as $row ) {
                if ( isset( $stats[ $row->event_type ] ) ) {
                    $stats[ $row->event_type ] = (int) $row->cnt;
                }
            }
        }

        return $stats;
    }

    protected function get_recent_alert_events( $limit = 10 ) {
        global $wpdb;

        $table = $this->sanitize_table_name( $this->get_activity_log_table_name() );
        if ( empty( $table ) ) {
            return array();
        }

        $limit = absint( $limit );
        if ( $limit <= 0 ) {
            $limit = 10;
        } elseif ( $limit > 50 ) {
            $limit = 50;
        }

        $alert_types = array(
            'malware_detected',
            'file_integrity_changed',
            'login_failed',
            'login_lockout',
            'admin_created',
            'admin_deleted',
            'role_changed',
            'plugin_activated',
            'plugin_deactivated',
            'settings_changed',
        );

        $args_recent_alert = array_merge( array( $table ), $alert_types, array( $limit ) );

        $cache_key_recent_alert = 'aegisshield_dash_recent_alert_' . md5( wp_json_encode( array( $table, $alert_types, $limit ) ) );


        $rows = wp_cache_get( $cache_key_recent_alert, 'aegisshield' );


        if ( false === $rows ) {

		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct DB access is required for custom tables.
			$wpdb->prepare(
				"SELECT event_time, event_type, user_id, ip_address, message
				 FROM {$table}
				 WHERE event_type IN (%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)
				 ORDER BY event_time DESC
				 LIMIT %d",
				$alert_types[0] ?? '',
				$alert_types[1] ?? '',
				$alert_types[2] ?? '',
				$alert_types[3] ?? '',
				$alert_types[4] ?? '',
				$alert_types[5] ?? '',
				$alert_types[6] ?? '',
				$alert_types[7] ?? '',
				$alert_types[8] ?? '',
				$alert_types[9] ?? '',
				$alert_types[10] ?? '',
				$limit
			),
			OBJECT
		);

        	wp_cache_set( $cache_key_recent_alert, $rows, 'aegisshield', MINUTE_IN_SECONDS );


        }
        if ( ! $rows ) {
            return array();
        }

        return $rows;
    }

    protected function build_db_health_widget() {
        global $wpdb;

        $data = array(
            'total_size_human'          => '',
            'largest_table'             => '',
            'largest_table_size_human'  => '',
            'spike_count'               => 0,
            'prefix_status'             => '',
        );

        try {
            $module  = new AS_Module_DB_Tools( $this->plugin );
            $summary = $module->get_table_summary();
        } catch ( \Exception $e ) {
            return $data;
        }

        $tables = isset( $summary['tables'] ) ? $summary['tables'] : array();

        if ( empty( $tables ) ) {
            return $data;
        }

        $total_bytes   = 0;
        $largest_name  = '';
        $largest_bytes = 0;
        $spike_count   = 0;

        foreach ( $tables as $table ) {
            $name        = isset( $table['name'] ) ? $table['name'] : '';
            $total       = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
            $delta_bytes = isset( $table['delta_bytes'] ) ? (int) $table['delta_bytes'] : 0;

            $total_bytes += $total;

            if ( $total > $largest_bytes ) {
                $largest_bytes = $total;
                $largest_name  = $name;
            }

            if ( $delta_bytes > 0 ) {
                $spike_count++;
            }
        }

        $data['total_size_human']         = size_format( $total_bytes );
        $data['largest_table']            = $largest_name;
        $data['largest_table_size_human'] = $largest_bytes ? size_format( $largest_bytes ) : '';
        $data['spike_count']              = $spike_count;

        if ( 'wp_' === $wpdb->prefix ) {
            $data['prefix_status'] = __( 'Using default \"wp_\" prefix (consider changing).', 'aegisshield-security' );
        } else {
            $data['prefix_status'] = __( 'Using a custom table prefix.', 'aegisshield-security' );
        }

        return $data;
    }
}