<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\AS_Logger;

defined( 'ABSPATH' ) || exit;

class AS_Page_Hardening {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    protected function is_pro_active() {
        return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
    }

    protected function render_pro_locked_notice() {
        ?>
        <div class="notice notice-warning inline">
            <p>
                <strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong>
            </p>
            <p>
                <?php esc_html_e( 'Upgrade to unlock vulnerability scanning and role risk analytics as part of your security hardening toolkit.', 'aegisshield-security' ); ?>
            </p>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                    <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                </a>
            </p>
        </div>
        <?php
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $is_pro = $this->is_pro_active();

        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        $current_tab = isset( $_GET['tab'] )
            ? sanitize_key( wp_unslash( $_GET['tab'] ) )
            : 'basic';

        if ( ! in_array( $current_tab, array( 'basic', 'vuln', 'roles' ), true ) ) {
            $current_tab = 'basic';
        }

        $page_url = admin_url( 'admin.php?page=aegisshield-hardening' );

        $disable_file_edit      = (string) $settings->get( $section, 'disable_file_edit', 'off' );
        $enable_hardening       = (string) $settings->get( $section, 'enable_hardening', 'off' );
        $xmlrpc_behavior        = (string) $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );
        $block_user_enum        = (string) $settings->get( $section, 'block_user_enum_behavior', 'a' );
        $hide_wp_version        = (string) $settings->get( $section, 'hide_wp_version', 'off' );
        $disable_editor_screens = (string) $settings->get( $section, 'disable_editor_screens', 'off' );
        $enforce_passwords      = (string) $settings->get( $section, 'enforce_strong_passwords', 'off' );
        $password_min_length    = (int) $settings->get( $section, 'password_min_length', 12 );
        $require_upper          = (string) $settings->get( $section, 'password_require_uppercase', 'on' );
        $require_lower          = (string) $settings->get( $section, 'password_require_lowercase', 'on' );
        $require_num            = (string) $settings->get( $section, 'password_require_number', 'on' );
        $require_sym            = (string) $settings->get( $section, 'password_require_symbol', 'on' );

        $updated = false;

        $old_values = array(
            'enable_hardening'           => $enable_hardening,
            'disable_file_edit'          => $disable_file_edit,
            'disable_xmlrpc_behavior'    => $xmlrpc_behavior,
            'block_user_enum_behavior'   => $block_user_enum,
            'hide_wp_version'            => $hide_wp_version,
            'disable_editor_screens'     => $disable_editor_screens,
            'enforce_strong_passwords'   => $enforce_passwords,
            'password_min_length'        => $password_min_length,
            'password_require_uppercase' => $require_upper,
            'password_require_lowercase' => $require_lower,
            'password_require_number'    => $require_num,
            'password_require_symbol'    => $require_sym,
        );

        if (
            'basic' === $current_tab
            && isset( $_POST['aegisshield_hardening_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_hardening_nonce'] ) ),
                'aegisshield_hardening_save'
            )
        ) {

            // Master toggle handling with explicit consent gate.
            $posted_enable = isset( $_POST['enable_hardening'] ) ? 'on' : 'off';
            $was_enabled   = in_array( (string) $enable_hardening, array( 'on', '1', 'yes' ), true );
            $wants_enable  = in_array( (string) $posted_enable, array( 'on', '1', 'yes' ), true );

            // If the admin is turning Hardening ON, require explicit consent.
            if ( ! $was_enabled && $wants_enable ) {
                $consent = isset( $_POST['hardening_consent'] ) ? sanitize_text_field( wp_unslash( $_POST['hardening_consent'] ) ) : '';
                if ( $consent !== 'yes' ) {
                    // Consent not granted; keep hardening OFF and do not modify any other settings.
                    $posted_enable = 'off';
                    $wants_enable  = false;
                }
            }

            // Always persist the master toggle.
            $enable_hardening = $posted_enable;
            $settings->set( $section, 'enable_hardening', $enable_hardening );

            // If Hardening is OFF, we do not change any other settings (WordPress-friendly behavior).
            if ( ! $wants_enable ) {
                $settings->save();
                $updated = true;
            } else {

            $disable_file_edit      = isset( $_POST['disable_file_edit'] ) ? 'on' : 'off';
            $xmlrpc_behavior        = isset( $_POST['disable_xmlrpc_behavior'] )
                ? sanitize_text_field( wp_unslash( $_POST['disable_xmlrpc_behavior'] ) )
                : 'c';
            $block_user_enum        = isset( $_POST['block_user_enum_behavior'] )
                ? sanitize_text_field( wp_unslash( $_POST['block_user_enum_behavior'] ) )
                : 'a';
            $hide_wp_version        = isset( $_POST['hide_wp_version'] ) ? 'on' : 'off';
            $disable_editor_screens = isset( $_POST['disable_editor_screens'] ) ? 'on' : 'off';
            $enforce_passwords      = isset( $_POST['enforce_strong_passwords'] ) ? 'on' : 'off';
            $password_min_length    = isset( $_POST['password_min_length'] )
                ? absint( wp_unslash( $_POST['password_min_length'] ) )
                : 12;

            if ( $password_min_length < 1 ) {
                $password_min_length = 1;
            }

			$logger = $this->plugin->get_logger();

            $logger->log(
                'hardening_rule_failed',
                __( 'Hardening input corrected: password_min_length was below 1 and was coerced to 1.', 'aegisshield-security' ),
                'low',
                array(
                    'setting_key' => 'password_min_length',
                    'original'    =>  isset( $_POST['password_min_length'] ) ? absint( wp_unslash( $_POST['password_min_length'] ) ) : null,
                    'new'         => 1,
                    'user_id'     => get_current_user_id(),
                )
            );


            $require_upper = isset( $_POST['password_require_uppercase'] ) ? 'on' : 'off';
            $require_lower = isset( $_POST['password_require_lowercase'] ) ? 'on' : 'off';
            $require_num   = isset( $_POST['password_require_number'] ) ? 'on' : 'off';
            $require_sym   = isset( $_POST['password_require_symbol'] ) ? 'on' : 'off';

            foreach ( array(
                'disable_file_edit'          => $disable_file_edit,
                'disable_xmlrpc_behavior'    => $xmlrpc_behavior,
                'block_user_enum_behavior'   => $block_user_enum,
                'hide_wp_version'            => $hide_wp_version,
                'disable_editor_screens'     => $disable_editor_screens,
                'enforce_strong_passwords'   => $enforce_passwords,
                'password_min_length'        => $password_min_length,
                'password_require_uppercase' => $require_upper,
                'password_require_lowercase' => $require_lower,
                'password_require_number'    => $require_num,
                'password_require_symbol'    => $require_sym,
            ) as $k => $new_val ) {
                $old_val = isset( $old_values[ $k ] ) ? $old_values[ $k ] : null;
                if ( (string) $old_val === (string) $new_val ) {
                    continue;
                }

                $event_type = 'hardening_policy_changed';
                $severity   = 'medium';
                if ( in_array( $k, array( 'disable_file_edit', 'hide_wp_version', 'disable_editor_screens', 'enforce_strong_passwords' ), true ) ) {
                    if ( (string) $new_val === 'on' ) {
                        $event_type = 'hardening_enabled';
                        $severity   = 'low';
                    } else {
                        $event_type = 'hardening_disabled';
                        $severity   = 'high';
                    }
                }

                $logger->log(
                    $event_type,
					// translators: %s: setting key being changed.
                    sprintf( __( 'Hardening setting changed: %s', 'aegisshield-security' ), $k ),
                    $severity,
                    array(
                        'setting_key' => $k,
                        'old'         => $old_val,
                        'new'         => $new_val,
                        'user_id'     => get_current_user_id(),
                    )
                );
            }

            if ( ! in_array( $xmlrpc_behavior, array( 'a', 'b', 'c' ), true ) ) {
                $logger->log(
                    'hardening_rule_failed',
                    __( 'Hardening input corrected: invalid XML-RPC behavior value was coerced to safe default.', 'aegisshield-security' ),
                    'low',
                    array(
                        'setting_key' => 'disable_xmlrpc_behavior',
                        'original'    =>  isset( $_POST['disable_xmlrpc_behavior'] ) ? sanitize_text_field( wp_unslash( $_POST['disable_xmlrpc_behavior'] ) ) : null,
                        'new'         => 'c',
                        'user_id'     => get_current_user_id(),
                    )
                );
                $xmlrpc_behavior = 'c';
            }
            if ( ! in_array( $block_user_enum, array( 'a', 'b' ), true ) ) {
                $logger->log(
                    'hardening_rule_failed',
                    __( 'Hardening input corrected: invalid user enumeration behavior value was coerced to safe default.', 'aegisshield-security' ),
                    'low',
                    array(
                        'setting_key' => 'block_user_enum_behavior',
                        'original'    =>  isset( $_POST['block_user_enum_behavior'] ) ? sanitize_text_field( wp_unslash( $_POST['block_user_enum_behavior'] ) ) : null,
                        'new'         => 'a',
                        'user_id'     => get_current_user_id(),
                    )
                );
                $block_user_enum = 'a';
            }

            $settings->set( $section, 'disable_file_edit', $disable_file_edit );
            $settings->set( $section, 'disable_xmlrpc_behavior', $xmlrpc_behavior );
            $settings->set( $section, 'block_user_enum_behavior', $block_user_enum );
            $settings->set( $section, 'hide_wp_version', $hide_wp_version );
            $settings->set( $section, 'disable_editor_screens', $disable_editor_screens );
            $settings->set( $section, 'enforce_strong_passwords', $enforce_passwords );
            $settings->set( $section, 'password_min_length', $password_min_length );
            $settings->set( $section, 'password_require_uppercase', $require_upper );
            $settings->set( $section, 'password_require_lowercase', $require_lower );
            $settings->set( $section, 'password_require_number', $require_num );
            $settings->set( $section, 'password_require_symbol', $require_sym );

            $settings->save();
            $updated = true;
            } // end else (wants_enable)
        }


        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Hardening', 'aegisshield-security' ); ?></h1>

            <p class="description">
                <?php esc_html_e(
                    'AegisShield Hardening strengthens WordPress by closing common security gaps while giving administrators full visibility into how their site is protected. Instead of silently blocking threats, it enforces proven security controls and records every violation with clear context, allowing admins to monitor, alert, and respond based on their own security policies. The result is a hardened WordPress environment that is proactive, transparent, and fully under the administrator’s control.',
                    'aegisshield-security'
                ); ?>
            </p>

            <?php if ( $updated && 'basic' === $current_tab ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Hardening settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( ! in_array( (string) $enable_hardening, array( 'on', '1', 'yes' ), true ) ) : ?>
                <div class="notice notice-warning inline">
                    <p>
                        <strong><?php esc_html_e( 'Hardening is currently disabled.', 'aegisshield-security' ); ?></strong>
                        <?php esc_html_e( 'Enable it only if you are proficient with WordPress. Some hardening options can affect login, API access, or editing behavior.', 'aegisshield-security' ); ?>
                    </p>
                </div>
            <?php endif; ?>

            <?php
            $basic_url = add_query_arg( 'tab', 'basic', $page_url );
            $vuln_url  = add_query_arg( 'tab', 'vuln',  $page_url );
            $roles_url = add_query_arg( 'tab', 'roles', $page_url );
            ?>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( $basic_url ); ?>"
                   class="nav-tab <?php echo esc_attr( ( 'basic' === $current_tab ) ? 'nav-tab-active' : '' ); ?>">
                    <?php esc_html_e( 'Basic Hardening', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( $vuln_url ); ?>"
                   class="nav-tab <?php echo ( 'vuln' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Vulnerability Scan', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'PRO', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>
                </a>
                <a href="<?php echo esc_url( $roles_url ); ?>"
                   class="nav-tab <?php echo ( 'roles' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Role Risk Analyzer', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
                            <?php esc_html_e( 'PRO', 'aegisshield-security' ); ?>
                        </span>
                    <?php endif; ?>
                </a>
            </h2>

            <div class="aegisshield-tab-panel">
                <?php if ( 'basic' === $current_tab ) : ?>
					<?php $hardening_is_on = in_array( (string) $enable_hardening, array( 'on', '1', 'yes' ), true ); ?>
					<?php $this->render_hardening_dashboard_visuals(); ?>
                    <h2><?php esc_html_e( 'Core Hardening Options', 'aegisshield-security' ); ?></h2>

					<form method="post">
						<?php
						wp_nonce_field(
							'aegisshield_hardening_save',
							'aegisshield_hardening_nonce'
						);
						?>

						<input type="hidden" name="hardening_consent" id="hardening_consent" value="" />

						<div class="notice notice-warning inline" style="margin: 12px 0 16px;">
							<p style="margin: 6px 0;">
								<strong><?php esc_html_e( 'Important:', 'aegisshield-security' ); ?></strong>
								<?php esc_html_e( 'Hardening can change WordPress default behaviors and may lock out users if misconfigured. It is completely OFF by default and should only be enabled by admins who are proficient with WordPress.', 'aegisshield-security' ); ?>
							</p>
						</div>

						<table class="form-table" role="presentation" style="margin-top:0;">
							<tbody>
								<tr>
									<th scope="row">
										<label for="enable_hardening">
											<?php esc_html_e( 'Enable Hardening', 'aegisshield-security' ); ?>
										</label>
									</th>
									<td>
										<label>
											<input type="checkbox" id="enable_hardening" name="enable_hardening" <?php checked( $hardening_is_on, true ); ?> />
											<?php esc_html_e( 'Turn on AegisShield Hardening controls.', 'aegisshield-security' ); ?>
										</label>
										<p class="description">
											<?php esc_html_e( 'When disabled, Hardening makes no changes to WordPress behavior.', 'aegisshield-security' ); ?>
										</p>
									</td>
								</tr>
							</tbody>
						</table>

						<script>
						(function () {
							function toggleHardeningControls(form) {
								var master = form.querySelector('#enable_hardening');
								if (!master) return;
								var enabled = !!master.checked;
								var fields = form.querySelectorAll('input, select, textarea, button');
								fields.forEach(function (el) {
									if (!el || !el.name) return;
									// Always allow nonce fields, master toggle, consent field, and submit buttons.
									if (el.id === 'enable_hardening') return;
									if (el.id === 'hardening_consent') return;
									if (el.name === 'aegisshield_hardening_nonce') return;
									if (el.name === '_wpnonce') return;
									if (el.name === '_wp_http_referer') return;
									if (el.type === 'submit') return;
									// Disable everything else when master toggle is OFF.
									el.disabled = !enabled;
								});
							}

							function bindConsentGate(form) {
								var master = form.querySelector('#enable_hardening');
								var consent = form.querySelector('#hardening_consent');
								if (!master || !consent) return;

								// Track initial state (server-rendered)
								var initiallyEnabled = <?php echo $hardening_is_on ? 'true' : 'false'; ?>;

								form.addEventListener('submit', function (e) {
									var enablingNow = (master.checked && !initiallyEnabled);
									if (enablingNow) {
										var msg =
											"You must be proficient with WordPress before turning this feature on.\n\n" +
											"It may lock out users and change WordPress default security measures.\n\n" +
											"Do you agree and want to enable Hardening?";
										if (!window.confirm(msg)) {
											// User did not agree -> leave it OFF.
											e.preventDefault();
											e.stopPropagation();
											master.checked = false;
											consent.value = '';
											toggleHardeningControls(form);
											return false;
										}
										consent.value = 'yes';
									}
								}, true);

								master.addEventListener('change', function () {
									// Reset consent if they toggle off/on before saving.
									if (!master.checked) {
										consent.value = '';
									}
									toggleHardeningControls(form);
								});

								toggleHardeningControls(form);
							}

							document.addEventListener('DOMContentLoaded', function () {
								var form = document.querySelector('form[method="post"]');
								if (!form) return;
								bindConsentGate(form);
							});
						})();
						</script>

						<script>
						(function () {
							// Only bind once
							if (window.__aegis_hardening_xmlrpc_warn_bound) return;
							window.__aegis_hardening_xmlrpc_warn_bound = true;

							document.addEventListener('DOMContentLoaded', function () {
								var form = document.querySelector('form[method="post"]');
								if (!form) return;

								form.addEventListener('submit', function (e) {
									var sel = document.getElementById('disable_xmlrpc_behavior');
									if (!sel) return;

									// Only warn when setting is "a" (deny with error page)
									if (sel.value === 'a') {
										var msg =
											"********************* WARNING ***************************\n\n" +
											"By enabling \"Deny XML-RPC requests with error page\", Wordpress Admin will no longer be accessible unless you turn off XML-RPC through SQL.  You will need to contact Aegisify Support in the future if you need WordPress Admin again.  We recommend leaving it on \"Allow core XML-RPC but restrict dangerous methods\".\n\n" +
											"If you still want too proceed, please click OK to proceed or Cancel to go back.";

										// OK = proceed (allow submit); Cancel = go back (prevent submit)
										if (!window.confirm(msg)) {
											e.preventDefault();
											e.stopPropagation();
											// keep them on the page + focus the dropdown
											try { sel.focus(); } catch (err) {}
											return false;
										}
									}
								}, true);
							});
						})();
						</script>

                        <table class="form-table" role="presentation">
                            <tbody>
                                <tr>
                                    <th scope="row">
                                        <label for="disable_file_edit">
                                            <?php esc_html_e( 'Disable file editor', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="disable_file_edit"
                                                   name="disable_file_edit"
                                                <?php checked( $disable_file_edit, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Prevent plugins & themes from being edited via the WordPress dashboard.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Recommended for most sites to reduce the impact if an admin account is compromised.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="disable_xmlrpc_behavior">
                                            <?php esc_html_e( 'XML-RPC behavior', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <select id="disable_xmlrpc_behavior"
                                                name="disable_xmlrpc_behavior">
                                            <option value="c" <?php selected( $xmlrpc_behavior, 'c' ); ?>>
                                                <?php esc_html_e(
                                                    'Allow core XML-RPC but restrict dangerous methods (recommended)',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="a" <?php selected( $xmlrpc_behavior, 'a' ); ?>>
                                                <?php esc_html_e(
                                                    'Deny XML-RPC requests with error page',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="b" <?php selected( $xmlrpc_behavior, 'b' ); ?>>
                                                <?php esc_html_e(
                                                    'Return HTTP 403 for XML-RPC (silent block)',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                        </select>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'XML-RPC is used by some apps and Jetpack. If you are unsure, keep the default safe option.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="block_user_enum_behavior">
                                            <?php esc_html_e( 'Block user enumeration', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <select id="block_user_enum_behavior"
                                                name="block_user_enum_behavior">
                                            <option value="a" <?php selected( $block_user_enum, 'a' ); ?>>
                                                <?php esc_html_e(
                                                    'Redirect ?author=N URLs to homepage',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                            <option value="b" <?php selected( $block_user_enum, 'b' ); ?>>
                                                <?php esc_html_e(
                                                    'Return 404 for user enumeration attempts',
                                                    'aegisshield-security'
                                                ); ?>
                                            </option>
                                        </select>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Prevents attackers from discovering usernames via author archives.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="hide_wp_version">
                                            <?php esc_html_e( 'Hide WordPress version', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="hide_wp_version"
                                                   name="hide_wp_version"
                                                <?php checked( $hide_wp_version, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Remove WordPress version from page output & assets.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'This makes it slightly harder for automated tools to fingerprint your site.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <label for="disable_editor_screens">
                                            <?php esc_html_e( 'Disable plugin & theme editor screens', 'aegisshield-security' ); ?>
                                        </label>
                                    </th>
                                    <td>
                                        <label>
                                            <input type="checkbox"
                                                   id="disable_editor_screens"
                                                   name="disable_editor_screens"
                                                <?php checked( $disable_editor_screens, 'on' ); ?> />
                                            <?php esc_html_e(
                                                'Hide the built-in plugin and theme code editor screens.',
                                                'aegisshield-security'
                                            ); ?>
                                        </label>
                                        <p class="description">
                                            <?php esc_html_e(
                                                'Use SFTP or version control instead of editing PHP files directly from the dashboard.',
                                                'aegisshield-security'
                                            ); ?>
                                        </p>
                                    </td>
                                </tr>

                                <tr>
                                    <th scope="row">
                                        <?php esc_html_e( 'Enforce strong passwords', 'aegisshield-security' ); ?>
                                    </th>
                                    <td>
                                        <fieldset>
                                            <label>
                                                <input type="checkbox"
                                                       name="enforce_strong_passwords"
                                                    <?php checked( $enforce_passwords, 'on' ); ?> />
                                                <?php esc_html_e(
                                                    'Require strong passwords for new and updated accounts.',
                                                    'aegisshield-security'
                                                ); ?>
                                            </label>

                                            <p class="description">
                                                <?php esc_html_e(
                                                    'When enabled, password strength checks will be enforced on the server.',
                                                    'aegisshield-security'
                                                ); ?>
                                            </p>

                                            <table>
                                                <tr>
                                                    <td>
                                                        <label for="password_min_length">
                                                            <?php esc_html_e(
                                                                'Minimum length',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                    <td>
                                                        <input type="number"
                                                               id="password_min_length"
                                                               name="password_min_length"
                                                               min="1"
                                                               value="<?php echo esc_attr( $password_min_length ); ?>" />
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_uppercase"
                                                                <?php checked( $require_upper, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one uppercase letter',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_lowercase"
                                                                <?php checked( $require_lower, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one lowercase letter',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_number"
                                                                <?php checked( $require_num, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one number',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td colspan="2">
                                                        <label>
                                                            <input type="checkbox"
                                                                   name="password_require_symbol"
                                                                <?php checked( $require_sym, 'on' ); ?> />
                                                            <?php esc_html_e(
                                                                'Require at least one symbol (e.g. !, @, #)',
                                                                'aegisshield-security'
                                                            ); ?>
                                                        </label>
                                                    </td>
                                                </tr>
                                            </table>
                                        </fieldset>
                                    </td>
                                </tr>
                            </tbody>
                        </table>

                        <?php submit_button( __( 'Save Changes', 'aegisshield-security' ) ); ?>
                    </form>

                <?php elseif ( 'vuln' === $current_tab ) : ?>

                    <?php
                    if ( ! $is_pro ) {
                        $this->render_pro_locked_notice();
                    } else {
                        if ( class_exists( '\AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Vuln' ) ) {
                            \AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Vuln::render_tab( $this->plugin );
                        } else {
                            echo '<p>' . esc_html__(
                                'Vulnerability Scan tab will be available in a future release.',
                                'aegisshield-security'
                            ) . '</p>';
                        }
                    }
                    ?>

                <?php elseif ( 'roles' === $current_tab ) : ?>

                    <?php
                    if ( ! $is_pro ) {
                        $this->render_pro_locked_notice();
                    } else {
                        if ( class_exists( '\AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Roles' ) ) {
                            \AegisShield\Admin_Pages\Hardening\AS_Page_Hardening_Tab_Roles::render_tab( $this->plugin );
                        } else {
                            echo '<p>' . esc_html__(
                                'Role Risk Analyzer tab will be available in a future release.',
                                'aegisshield-security'
                            ) . '</p>';
                        }
                    }
                    ?>

                <?php endif; ?>
            </div>
        </div>
        <?php
    }
	/**
	 * Hardening Dashboard Visuals (4 inline charts) under Basic Hardening tab.
	 * Uses ONLY Hardening module data (settings + stored scan outputs).
	 */
	protected function render_hardening_dashboard_visuals() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data = $this->get_hardening_dashboard_data();

		// Use local Chart.js already shipped in plugin (no external/CDN).
		$plugin_file = defined( 'AEGISSHIELD_PLUGIN_FILE' )
			? AEGISSHIELD_PLUGIN_FILE
			: dirname( __FILE__, 3 ) . '/aegisshield-security.php';

		$chart_src = plugins_url( 'assets/js/chart.umd.min.js', $plugin_file );
		?>
		<style>
			.aegis-hard-dash {
				display: grid;
				grid-template-columns: repeat(4, minmax(0, 1fr));
				gap: 12px;
				margin: 12px 0 14px;
			}
			.aegis-hard-card {
				background: #fff;
				border: 1px solid #dcdcde;
				border-radius: 10px;
				padding: 12px 12px 10px;
				box-shadow: 0 1px 1px rgba(0,0,0,.04);
				min-height: 235px;
			}
			.aegis-hard-card h3 {
				margin: 0 0 8px;
				font-size: 13px;
				line-height: 1.3;
				display: flex;
				align-items: center;
				gap: 8px;
			}
			.aegis-hard-pill {
				font-size: 12px;
				padding: 2px 8px;
				border-radius: 999px;
				border: 1px solid #dcdcde;
				background: #f6f7f7;
			}
			.aegis-hard-canvas { position: relative; height: 175px; }
			.aegis-hard-sub {
				margin-top: 8px;
				font-size: 12px;
				color: #646970;
				display: flex;
				justify-content: space-between;
				gap: 10px;
			}
			@media (max-width: 1200px) { .aegis-hard-dash { grid-template-columns: repeat(2, minmax(0, 1fr)); } }
			@media (max-width: 782px)  { .aegis-hard-dash { grid-template-columns: 1fr; } }
		</style>

		<div class="aegis-hard-dash" id="aegis-hard-dash"
			 data-hard="<?php echo esc_attr( wp_json_encode( $data ) ); ?>">

			<div class="aegis-hard-card">
				<h3>
					<span class="dashicons dashicons-shield-alt"></span>
					<?php esc_html_e( 'Hardening Coverage', 'aegisshield-security' ); ?>
					<span class="aegis-hard-pill">
						<?php echo esc_html( (string) $data['coverage']['secured'] ); ?>/<?php echo esc_html( (string) $data['coverage']['total'] ); ?>
					</span>
				</h3>
				<div class="aegis-hard-canvas"><canvas id="aegisHardChartCoverage"></canvas></div>
				<div class="aegis-hard-sub">
					<span><?php esc_html_e( 'Secured vs Needs Attention', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['coverage']['needs_attention'] ); ?></span>
				</div>
			</div>

			<div class="aegis-hard-card">
				<h3>
					<span class="dashicons dashicons-warning"></span>
					<?php esc_html_e( 'Vulnerability Findings', 'aegisshield-security' ); ?>
					<span class="aegis-hard-pill">
						<?php echo esc_html( (string) $data['vuln']['total'] ); ?>
					</span>
				</h3>
				<div class="aegis-hard-canvas"><canvas id="aegisHardChartVulnType"></canvas></div>
				<div class="aegis-hard-sub">
					<span><?php esc_html_e( 'Plugins vs Themes (by severity)', 'aegisshield-security' ); ?></span>
					<span>
						<?php echo esc_html( $data['vuln']['last_run_human'] ); ?>
					</span>
				</div>
			</div>

			<div class="aegis-hard-card">
				<h3>
					<span class="dashicons dashicons-chart-bar"></span>
					<?php esc_html_e( 'Top Risk Components', 'aegisshield-security' ); ?>
					<span class="aegis-hard-pill"><?php echo esc_html( (string) count( $data['vuln']['top']['labels'] ) ); ?></span>
				</h3>
				<div class="aegis-hard-canvas"><canvas id="aegisHardChartTop"></canvas></div>
				<div class="aegis-hard-sub">
					<span><?php esc_html_e( 'Highest risk plugins/themes', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['vuln']['counts']['high'] ); ?> <?php esc_html_e('high', 'aegisshield-security'); ?></span>
				</div>
			</div>

			<div class="aegis-hard-card">
				<h3>
					<span class="dashicons dashicons-admin-users"></span>
					<?php esc_html_e( 'Role Risk Exposure', 'aegisshield-security' ); ?>
					<span class="aegis-hard-pill"><?php echo esc_html( (string) $data['roles']['total_flagged'] ); ?></span>
				</h3>
				<div class="aegis-hard-canvas"><canvas id="aegisHardChartCaps"></canvas></div>
				<div class="aegis-hard-sub">
					<span><?php esc_html_e( 'Dangerous capabilities among risky users', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['roles']['counts']['high'] ); ?> <?php esc_html_e('high', 'aegisshield-security'); ?></span>
				</div>
			</div>

		</div>

		<script>
		(function(){
			// Load chart script once.
			if (typeof window.__aegisChartLoaded === 'undefined') {
				window.__aegisChartLoaded = true;
				var s=document.createElement('script');
				s.src=<?php echo wp_json_encode( $chart_src ); ?>;
				s.onload=initHardCharts;
				document.head.appendChild(s);
			} else {
				initHardCharts();
			}

			function initHardCharts() {
				var wrap = document.getElementById('aegis-hard-dash');
				if (!wrap) return;
				var raw = wrap.getAttribute('data-hard');
				if (!raw) return;

				var data;
				try { data = JSON.parse(raw); } catch(e){ return; }
				if (typeof Chart === 'undefined') return;

				function ctx(id){ var el=document.getElementById(id); return el ? el.getContext('2d') : null; }

				// 1) Coverage (doughnut)
				var c1 = ctx('aegisHardChartCoverage');
				if (c1 && data.coverage) {
					new Chart(c1, {
						type: 'doughnut',
						data: {
							labels: ['Secured', 'Needs Attention'],
							datasets: [{ data: [data.coverage.secured, data.coverage.needs_attention] }]
						},
						options: {
							responsive: true,
							maintainAspectRatio: false,
							plugins: { legend: { position: 'bottom' } },
							cutout: '65%'
						}
					});
				}

				// 2) Vuln by type (stacked bar): Plugins/Themes with High/Medium/Low
				var c2 = ctx('aegisHardChartVulnType');
				if (c2 && data.vuln && data.vuln.type) {
					new Chart(c2, {
						type: 'bar',
						data: {
							labels: ['Plugins', 'Themes'],
							datasets: [
								{ label: 'High',   data: [data.vuln.type.plugins.high,   data.vuln.type.themes.high],   stack: 'sev' },
								{ label: 'Medium', data: [data.vuln.type.plugins.medium, data.vuln.type.themes.medium], stack: 'sev' },
								{ label: 'Low',    data: [data.vuln.type.plugins.low,    data.vuln.type.themes.low],    stack: 'sev' }
							]
						},
						options: {
							responsive: true,
							maintainAspectRatio: false,
							plugins: { legend: { position: 'bottom' } },
							scales: {
								x: { stacked: true, grid: { display: false } },
								y: { stacked: true, beginAtZero: true, ticks: { precision: 0 } }
							}
						}
					});
				}

				// 3) Top risk components (horizontal bar)
				var c3 = ctx('aegisHardChartTop');
				if (c3 && data.vuln && data.vuln.top && data.vuln.top.labels) {
					new Chart(c3, {
						type: 'bar',
						data: {
							labels: data.vuln.top.labels,
							datasets: [{ label: 'Severity score', data: data.vuln.top.scores }]
						},
						options: {
							indexAxis: 'y',
							responsive: true,
							maintainAspectRatio: false,
							plugins: { legend: { display: false } },
							scales: {
								x: { beginAtZero: true, ticks: { precision: 0 } },
								y: { grid: { display: false } }
							}
						}
					});
				}

				// 4) Dangerous caps exposure (radar)
				var c4 = ctx('aegisHardChartCaps');
				if (c4 && data.roles && data.roles.caps) {
					new Chart(c4, {
						type: 'radar',
						data: {
							labels: data.roles.caps.labels,
							datasets: [{
								label: 'Count',
								data: data.roles.caps.values,
								fill: true
							}]
						},
						options: {
							responsive: true,
							maintainAspectRatio: false,
							plugins: { legend: { position: 'bottom' } },
							scales: { r: { beginAtZero: true, ticks: { precision: 0 } } }
						}
					});
				}
			}
		})();
		</script>
		<?php
	}

	/**
	 * Builds dashboard data using ONLY Hardening module settings + stored scan outputs.
	 */
	protected function get_hardening_dashboard_data() {
		$settings = $this->plugin->get_settings();
		$section  = 'hardening';

		// Basic hardening coverage (ONLY things on Basic tab + safe behaviors).
		$items = [];

		$items['disable_file_edit']      = (string) $settings->get( $section, 'disable_file_edit', 'off' ) === 'on';
		$items['hide_wp_version']        = (string) $settings->get( $section, 'hide_wp_version', 'off' ) === 'on';
		$items['disable_editor_screens'] = (string) $settings->get( $section, 'disable_editor_screens', 'off' ) === 'on';
		$items['enforce_strong_passwords'] = (string) $settings->get( $section, 'enforce_strong_passwords', 'off' ) === 'on';

		// XML-RPC behavior: a = deny with error page (dangerous lockout), b/c = acceptable (c recommended).
		$xmlrpc = (string) $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );
		$items['xmlrpc_safe_mode'] = in_array( $xmlrpc, ['b','c'], true );

		// User enum behavior: b = 404 is strongest, a = redirect is weaker.
		$enum = (string) $settings->get( $section, 'block_user_enum_behavior', 'a' );
		$items['user_enum_strong'] = ( $enum === 'b' );

		$total = count( $items );
		$secured = 0;
		foreach ( $items as $ok ) { if ( $ok ) { $secured++; } }
		$needs_attention = max( 0, $total - $secured );

		// Vulnerability results
		$v = $settings->get( $section, 'hardening_vuln_results', [] );
		if ( ! is_array( $v ) ) { $v = []; }

		$plugins = isset( $v['plugins'] ) && is_array( $v['plugins'] ) ? $v['plugins'] : [];
		$themes  = isset( $v['themes'] )  && is_array( $v['themes'] )  ? $v['themes']  : [];

		$sev_counts = [
			'plugins' => ['high'=>0,'medium'=>0,'low'=>0],
			'themes'  => ['high'=>0,'medium'=>0,'low'=>0],
		];

		$flat = []; // for top list

		$score = function( $risk ) {
			$risk = strtolower( (string) $risk );
			if ( $risk === 'high' ) return 3;
			if ( $risk === 'medium' ) return 2;
			return 1; // low/unknown treated as low
		};

		$acc = function( $type, $slug, $row ) use ( &$sev_counts, &$flat, $score ) {
			$risk = is_array( $row ) && isset( $row['risk'] ) ? $row['risk'] : 'low';
			$r = strtolower( (string) $risk );
			if ( ! in_array( $r, ['high','medium','low'], true ) ) { $r = 'low'; }
			$sev_counts[$type][$r]++;

			$flat[] = [
				'label' => ($type === 'plugins' ? 'P: ' : 'T: ') . (string) $slug,
				'score' => $score( $r ),
			];
		};

		foreach ( $plugins as $slug => $row ) { $acc('plugins', $slug, $row); }
		foreach ( $themes  as $slug => $row ) { $acc('themes',  $slug, $row); }

		usort( $flat, function($a,$b){ return $b['score'] <=> $a['score']; } );
		$flat = array_slice( $flat, 0, 6 );

		$top_labels = array_map( function($x){ return $x['label']; }, $flat );
		$top_scores = array_map( function($x){ return $x['score']; }, $flat );

		$last_run = (int) $settings->get( $section, 'hardening_vuln_last_run', 0 );
		// translators: %s: human-readable date/time of the last vulnerability scan run.
		$last_run_human = $last_run ? sprintf( __( 'Last run: %s', 'aegisshield-security' ), date_i18n( 'M j, Y g:ia', $last_run ) ) : __( 'Never run', 'aegisshield-security' );

		$total_vuln = array_sum( $sev_counts['plugins'] ) + array_sum( $sev_counts['themes'] );

		$vuln_counts = [
			'high'   => $sev_counts['plugins']['high']   + $sev_counts['themes']['high'],
			'medium' => $sev_counts['plugins']['medium'] + $sev_counts['themes']['medium'],
			'low'    => $sev_counts['plugins']['low']    + $sev_counts['themes']['low'],
		];

		// Role risk results
		$rr = $settings->get( $section, 'hardening_role_risk_results', [] );
		if ( ! is_array( $rr ) ) { $rr = []; }

		$role_counts = ['high'=>0,'medium'=>0];
		$cap_names = ['manage_options','edit_plugins','delete_users','unfiltered_html','edit_theme_options'];
		$cap_freq = array_fill_keys( $cap_names, 0 );

		foreach ( $rr as $row ) {
			if ( ! is_array( $row ) ) { continue; }
			$risk = strtolower( (string) ( $row['risk'] ?? '' ) );
			if ( $risk === 'high' ) { $role_counts['high']++; }
			elseif ( $risk === 'medium' ) { $role_counts['medium']++; }

			$caps = isset( $row['dangerous_caps'] ) && is_array( $row['dangerous_caps'] ) ? $row['dangerous_caps'] : [];
			foreach ( $caps as $c ) {
				$c = (string) $c;
				if ( isset( $cap_freq[$c] ) ) {
					$cap_freq[$c]++;
				}
			}
		}

		return [
			'coverage' => [
				'secured' => $secured,
				'needs_attention' => $needs_attention,
				'total' => $total,
			],
			'vuln' => [
				'total' => $total_vuln,
				'last_run_human' => $last_run_human,
				'type' => [
					'plugins' => $sev_counts['plugins'],
					'themes'  => $sev_counts['themes'],
				],
				'counts' => $vuln_counts,
				'top' => [
					'labels' => $top_labels,
					'scores' => $top_scores,
				],
			],
			'roles' => [
				'total_flagged' => (int) count( $rr ),
				'counts' => $role_counts,
				'caps' => [
					'labels' => array_keys( $cap_freq ),
					'values' => array_values( $cap_freq ),
				],
			],
		];
	}

}