<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\AS_Logger;

defined( 'ABSPATH' ) || exit;

class AS_Page_Login_Guard {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }


    protected function is_pro_active() {
        return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
    }

    protected function render_pro_locked_notice() {
        ?>
        <div class="notice notice-warning inline">
            <p>
                <strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong>
            </p>
            <p>
                <?php esc_html_e( 'Upgrade to unlock advanced login protection features including MFA, trusted devices, adaptive lockouts, geo rules, and honeypot login traps.', 'aegisshield-security' ); ?>
            </p>
            <p>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>" class="button button-primary">
                    <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
                </a>
            </p>
        </div>
        <?php
    }

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }

        $active_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'overview';

        if ( ! in_array( $active_tab, array( 'overview', 'auth', 'lockouts', 'geo' ), true ) ) {
            $active_tab = 'overview';
        }

        $is_pro = $this->is_pro_active();

        // Handle saves BEFORE rendering
        if ( isset( $_POST['aegisshield_login_guard_save_overview'] ) ) {
            check_admin_referer( 'aegisshield_login_guard_overview' );
            $this->save_overview_tab();

        } elseif ( isset( $_POST['aegisshield_login_guard_save_auth'] ) ) {
            check_admin_referer( 'aegisshield_login_guard_auth' );

            if ( $is_pro ) {
                $this->save_authentication_tab();
            } else {
                wp_safe_redirect(
                    add_query_arg(
                        array(
                            'tab'                        => 'auth',
                            'settings-updated'           => '0',
                            'aegisshield_pro_required'   => '1',
                        )
                    )
                );
                exit;
            }

        } elseif ( isset( $_POST['aegisshield_login_guard_save_lockouts'] ) ) {
            check_admin_referer( 'aegisshield_login_guard_lockouts' );

            if ( $is_pro ) {
                $this->save_lockouts_tab();
            } else {
                wp_safe_redirect(
                    add_query_arg(
                        array(
                            'tab'                        => 'lockouts',
                            'settings-updated'           => '0',
                            'aegisshield_pro_required'   => '1',
                        )
                    )
                );
                exit;
            }

        } elseif ( isset( $_POST['aegisshield_login_guard_save_geo'] ) ) {
            check_admin_referer( 'aegisshield_login_guard_geo' );

            if ( $is_pro ) {
                $this->save_geo_tab();
            } else {
                wp_safe_redirect(
                    add_query_arg(
                        array(
                            'tab'                        => 'geo',
                            'settings-updated'           => '0',
                            'aegisshield_pro_required'   => '1',
                        )
                    )
                );
                exit;
            }

        } elseif ( isset( $_POST['aegisshield_unlock_ip'] ) ) {
            check_admin_referer( 'aegisshield_unlock_ip' );
            $this->handle_unlock_ip();
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Login Guard', 'aegisshield-security' ); ?></h1>
			<p class="description">
                <?php esc_html_e( 'Login Guard is a comprehensive WordPress login security layer that protects against brute-force attacks, automated bots, and account takeovers by enforcing smart lockouts, MFA, device trust, and behavioral rules at the authentication level. It actively monitors, enforces, and logs suspicious login activity so administrators gain both real-time protection and clear visibility into how and why access attempts are allowed or blocked.', 'aegisshield-security' ); ?>
            </p>

            <?php if ( isset( $_GET['settings-updated'] ) && '1' === (string) sanitize_text_field( wp_unslash( $_GET['settings-updated'] ) ) ) : ?>
                <div id="message" class="updated notice is-dismissible">
                    <p><?php esc_html_e( 'Settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( isset( $_GET['aegisshield_pro_required'] ) && '1' === (string) sanitize_text_field( wp_unslash( $_GET['aegisshield_pro_required'] ) ) ) : ?>
                <div class="notice notice-warning is-dismissible">
                    <p><?php esc_html_e( 'These settings are available in AegisShield Pro. Your changes were not saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'overview' ) ) ); ?>" class="nav-tab <?php echo ( 'overview' === $active_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Overview', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'auth' ) ) ); ?>" class="nav-tab <?php echo ( 'auth' === $active_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Authentication (MFA & Devices)', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px; font-size:11px; padding:2px 5px; border-radius:3px; background:#ff9800; color:#fff; text-transform:uppercase;"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
                    <?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'lockouts' ) ) ); ?>" class="nav-tab <?php echo ( 'lockouts' === $active_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Lockouts & Rate Limits', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px; font-size:11px; padding:2px 5px; border-radius:3px; background:#ff9800; color:#fff; text-transform:uppercase;"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
                    <?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( array( 'tab' => 'geo' ) ) ); ?>" class="nav-tab <?php echo ( 'geo' === $active_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Geo Rules & Honeypot', 'aegisshield-security' ); ?>
                    <?php if ( ! $is_pro ) : ?>
                        <span class="aegis-badge-pro" style="margin-left:4px; font-size:11px; padding:2px 5px; border-radius:3px; background:#ff9800; color:#fff; text-transform:uppercase;"><?php esc_html_e( 'Pro', 'aegisshield-security' ); ?></span>
                    <?php endif; ?>
                </a>
            </h2>

        <?php

        switch ( $active_tab ) {
            case 'auth':
                $this->render_authentication_tab();
                break;

            case 'lockouts':
                $this->render_lockouts_tab();
                break;

            case 'geo':
                $this->render_geo_tab();
                break;

            case 'overview':
            default:
                $this->render_overview_tab();
                break;
        }

        echo '</div>';
    }

    protected function get_login_guard_settings() {
        $settings_obj = $this->plugin->get_settings();

        $keys = array(
            'max_attempts',
            'lockout_minutes',
            'honeypot_enabled',
            'block_unknown_user',

            'pro_2fa_enabled',
            'pro_2fa_methods',
            'pro_2fa_enforced_roles',
            'pro_mfa_qr_engine',

            'pro_trusted_devices_enabled',
            'pro_trusted_device_lifetime_days',

            'pro_lockout_known_threshold',
            'pro_lockout_unknown_threshold',
            'pro_lockout_admin_threshold',
            'pro_lockout_progressive_enabled',
            'pro_username_enum_detection_enabled',

            'pro_geo_mode',
            'pro_allowed_countries',

            'pro_honeypot_enabled',
            'pro_honeypot_slug',
        );

        $settings = array();
        foreach ( $keys as $key ) {
            $settings[ $key ] = $settings_obj->get( 'login_guard', $key, null );
        }

        return $settings;
    }

    /* ================================================================
     *  OVERVIEW TAB
     * ================================================================ */

    protected function render_overview_tab() {
        $settings = $this->get_login_guard_settings();
        $max_attempts = isset( $settings['max_attempts'] ) ? (int) $settings['max_attempts'] : 5;
        $lockout_mins = isset( $settings['lockout_minutes'] ) ? (int) $settings['lockout_minutes'] : 15;
        $honeypot     = ! empty( $settings['honeypot_enabled'] );
        $block_unknown = ! empty( $settings['block_unknown_user'] );
        ?>
		<?php $this->render_login_guard_dashboard_visuals(); ?>
        <form method="post">
            <?php wp_nonce_field( 'aegisshield_login_guard_overview' ); ?>
            <input type="hidden" name="aegisshield_login_guard_save_overview" value="1" />

            <h2><?php esc_html_e( 'Baseline Login Protection', 'aegisshield-security' ); ?></h2>

            <table class="form-table">
                <tr>
                    <th><label for="aegis_max_attempts"><?php esc_html_e( 'Max failed attempts', 'aegisshield-security' ); ?></label></th>
                    <td>
                        <input type="number" name="max_attempts" id="aegis_max_attempts" class="small-text" min="1"
                            value="<?php echo esc_attr( $max_attempts ); ?>" />
                    </td>
                </tr>

                <tr>
                    <th><label for="aegis_lockout_minutes"><?php esc_html_e( 'Lockout duration (minutes)', 'aegisshield-security' ); ?></label></th>
                    <td>
                        <input type="number" name="lockout_minutes" id="aegis_lockout_minutes" class="small-text" min="1"
                            value="<?php echo esc_attr( $lockout_mins ); ?>" />
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Block unknown usernames', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="block_unknown_user" value="1" <?php checked( $block_unknown ); ?> /> Enable</label>
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Honeypot login field', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="honeypot_enabled" value="1" <?php checked( $honeypot ); ?> /> Enable</label>
                    </td>
                </tr>
            </table>

            <?php submit_button( __( 'Save Login Guard Settings', 'aegisshield-security' ) ); ?>
        </form>
        <?php
    }

	protected function render_login_guard_dashboard_visuals() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data = $this->get_login_guard_dashboard_data();

		$plugin_file = defined( 'AEGISSHIELD_PLUGIN_FILE' )
			? AEGISSHIELD_PLUGIN_FILE
			: dirname( __FILE__, 3 ) . '/aegisshield-security.php';

		$chart_src = plugins_url( 'assets/js/chart.umd.min.js', $plugin_file );

		?>
		<style>
			.aegis-lg-dash {
				display: grid;
				grid-template-columns: repeat(4, minmax(0, 1fr));
				gap: 12px;
				margin: 14px 0 10px;
			}
			.aegis-lg-card {
				background: #fff;
				border: 1px solid #dcdcde;
				border-radius: 10px;
				padding: 12px 12px 10px;
				box-shadow: 0 1px 1px rgba(0,0,0,.04);
				min-height: 220px;
			}
			.aegis-lg-card h3 {
				margin: 0 0 8px;
				font-size: 13px;
				line-height: 1.3;
				display: flex;
				align-items: center;
				gap: 8px;
			}
			.aegis-lg-pill {
				font-size: 12px;
				padding: 2px 8px;
				border-radius: 999px;
				border: 1px solid #dcdcde;
				background: #f6f7f7;
			}
			.aegis-lg-canvas {
				position: relative;
				height: 160px;
			}
			.aegis-lg-sub {
				margin-top: 8px;
				font-size: 12px;
				color: #646970;
				display: flex;
				justify-content: space-between;
				gap: 10px;
			}

			@media (max-width: 1200px) {
				.aegis-lg-dash { grid-template-columns: repeat(2, minmax(0, 1fr)); }
			}
			@media (max-width: 782px) {
				.aegis-lg-dash { grid-template-columns: 1fr; }
			}
		</style>

		<div class="aegis-lg-dash" id="aegis-lg-dash"
			 data-lg="<?php echo esc_attr( wp_json_encode( $data ) ); ?>">
			<div class="aegis-lg-card">
				<h3>
					<span class="dashicons dashicons-chart-area"></span>
					<?php esc_html_e( 'Login Activity (Last 24 Hours)', 'aegisshield-security' ); ?>
					<span class="aegis-lg-pill"><?php echo esc_html( (string) $data['totals']['recent_24h'] ); ?></span>
				</h3>
				<div class="aegis-lg-canvas"><canvas id="aegisLgChartActivity"></canvas></div>
				<div class="aegis-lg-sub">
					<span><?php esc_html_e( 'IPs with recent attempts', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['totals']['recent_24h'] ); ?></span>
				</div>
			</div>

			<div class="aegis-lg-card">
				<h3>
					<span class="dashicons dashicons-shield-alt"></span>
					<?php esc_html_e( 'Top IPs by Attempts', 'aegisshield-security' ); ?>
					<span class="aegis-lg-pill"><?php echo esc_html( (string) $data['totals']['tracked_ips'] ); ?></span>
				</h3>
				<div class="aegis-lg-canvas"><canvas id="aegisLgChartTopIps"></canvas></div>
				<div class="aegis-lg-sub">
					<span><?php esc_html_e( 'Tracked IPs', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['totals']['tracked_ips'] ); ?></span>
				</div>
			</div>

			<div class="aegis-lg-card">
				<h3>
					<span class="dashicons dashicons-lock"></span>
					<?php esc_html_e( 'Lockout Status', 'aegisshield-security' ); ?>
					<span class="aegis-lg-pill"><?php echo esc_html( (string) $data['totals']['active_lockouts'] ); ?></span>
				</h3>
				<div class="aegis-lg-canvas"><canvas id="aegisLgChartLockouts"></canvas></div>
				<div class="aegis-lg-sub">
					<span><?php esc_html_e( 'Active lockouts', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['totals']['active_lockouts'] ); ?></span>
				</div>
			</div>

			<div class="aegis-lg-card">
				<h3>
					<span class="dashicons dashicons-admin-users"></span>
					<?php esc_html_e( 'Admin MFA Coverage', 'aegisshield-security' ); ?>
					<span class="aegis-lg-pill"><?php echo esc_html( (string) $data['mfa']['admins_total'] ); ?></span>
				</h3>
				<div class="aegis-lg-canvas"><canvas id="aegisLgChartMfa"></canvas></div>
				<div class="aegis-lg-sub">
					<span><?php esc_html_e( 'Admins with MFA enabled', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( (string) $data['mfa']['admins_enabled'] ); ?></span>
				</div>
			</div>
		</div>

		<?php
		wp_enqueue_script( 'aegisshield-login-guard-chartjs', $chart_src, array(), ( defined( 'AEGISSHIELD_VERSION' ) ? AEGISSHIELD_VERSION : '1.0.0' ), true );
		?>
		<script>
		(function(){
			var wrap = document.getElementById('aegis-lg-dash');
			if (!wrap) return;

			var raw = wrap.getAttribute('data-lg');
			if (!raw) return;

			var data;
			try { data = JSON.parse(raw); } catch(e){ return; }

			if (typeof Chart === 'undefined') return;

			function ctx(id){ var el=document.getElementById(id); return el ? el.getContext('2d') : null; }

			// 1) Activity (line/area)
			var c1 = ctx('aegisLgChartActivity');
			if (c1 && data.activity && data.activity.labels) {
				new Chart(c1, {
					type: 'line',
					data: {
						labels: data.activity.labels,
						datasets: [{
							label: 'IPs / hour',
							data: data.activity.values,
							fill: true,
							tension: 0.35,
							pointRadius: 0
						}]
					},
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: { legend: { display: false }, tooltip: { intersect: false } },
						scales: {
							x: { grid: { display: false } },
							y: { beginAtZero: true, ticks: { precision: 0 } }
						}
					}
				});
			}

			// 2) Top IPs (horizontal bar)
			var c2 = ctx('aegisLgChartTopIps');
			if (c2 && data.top_ips && data.top_ips.labels) {
				new Chart(c2, {
					type: 'bar',
					data: {
						labels: data.top_ips.labels,
						datasets: [{
							label: 'Attempts',
							data: data.top_ips.values
						}]
					},
					options: {
						indexAxis: 'y',
						responsive: true,
						maintainAspectRatio: false,
						plugins: { legend: { display: false } },
						scales: {
							x: { beginAtZero: true, ticks: { precision: 0 } },
							y: { grid: { display: false } }
						}
					}
				});
			}

			// 3) Lockouts (doughnut)
			var c3 = ctx('aegisLgChartLockouts');
			if (c3 && data.lockouts) {
				new Chart(c3, {
					type: 'doughnut',
					data: {
						labels: ['Active lockouts', 'Tracked (not locked)'],
						datasets: [{
							data: [data.lockouts.active, data.lockouts.not_locked]
						}]
					},
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: { legend: { position: 'bottom' } },
						cutout: '65%'
					}
				});
			}

			// 4) MFA coverage (doughnut)
			var c4 = ctx('aegisLgChartMfa');
			if (c4 && data.mfa) {
				var enabled = parseInt(data.mfa.admins_enabled || 0, 10);
				var total = parseInt(data.mfa.admins_total || 0, 10);
				var disabled = Math.max(0, total - enabled);

				new Chart(c4, {
					type: 'doughnut',
					data: {
						labels: ['MFA enabled', 'Not enabled'],
						datasets: [{
							data: [enabled, disabled]
						}]
					},
					options: {
						responsive: true,
						maintainAspectRatio: false,
						plugins: { legend: { position: 'bottom' } },
						cutout: '65%'
					}
				});
			}
		})();
		</script>
		<?php
	}


	protected function get_login_guard_dashboard_data() {
		global $wpdb;

        // Nonce verification for unlock action (admin POST).
        check_admin_referer( 'aegisshield_unlock_ip' );

		$table = $wpdb->prefix . 'aegisshield_login_attempts';
		$now   = current_time( 'mysql' );

		$cache_key = 'aegisshield_lg_dash_' . gmdate( 'YmdHi' );
		$cached    = wp_cache_get( $cache_key, 'aegisshield' );
		if ( is_array( $cached ) ) {
			return $cached;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; results cached via wp_cache_* above.
		$tracked_ips = (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM %i', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
		$active_lockouts = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
			$wpdb->prepare( 'SELECT COUNT(*) FROM %i WHERE locked_until > %s', $table, $now )
		);

		$not_locked = max( 0, $tracked_ips - $active_lockouts );

		// Activity: count IPs with last_attempt per hour, last 24 hours.
		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; results cached via wp_cache_* above.
			$wpdb->prepare(
				"SELECT DATE_FORMAT(last_attempt,'%%Y-%%m-%%d %%H:00:00') AS hour_bucket, COUNT(*) AS c
				 FROM %i
				 WHERE last_attempt >= (DATE_SUB(%s, INTERVAL 24 HOUR))
				 GROUP BY hour_bucket
				 ORDER BY hour_bucket ASC",
				$table,
				$now
			),
			ARRAY_A
		);

		$map = array();
		foreach ( (array) $rows as $r ) {
			$map[ (string) $r['hour_bucket'] ] = (int) $r['c'];
		}

		$activity_labels = array();
		$activity_values = array();
		$ts_now = strtotime( $now );
		// Build a full 24h series so chart never looks “broken” when there are gaps.
		for ( $i = 23; $i >= 0; $i-- ) {
			$t = $ts_now - ( $i * HOUR_IN_SECONDS );
			$bucket = gmdate( 'Y-m-d H:00:00', $t + ( get_option('gmt_offset') * HOUR_IN_SECONDS ) ); // align to WP tz-ish
			$label  = date_i18n( 'g a', strtotime( $bucket ) );
			$activity_labels[] = $label;
			$activity_values[] = isset( $map[ $bucket ] ) ? (int) $map[ $bucket ] : 0;
		}

		$recent_24h = array_sum( $activity_values );

		// Top IPs by attempts (current table state).
		$top = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; results cached via wp_cache_* above.
			$wpdb->prepare(
				"SELECT ip_address, attempts
				 FROM %i
				 ORDER BY attempts DESC
				 LIMIT 8",
				$table
			),
			ARRAY_A
		);

		$top_labels = array();
		$top_values = array();
		foreach ( (array) $top as $r ) {
			$top_labels[] = (string) $r['ip_address'];
			$top_values[] = (int) $r['attempts'];
		}

		// Admin MFA coverage: user_meta aegisshield_2fa_enabled.
		$admin_ids = get_users( array(
			'role'   => 'administrator',
			'fields' => array( 'ID' ),
		) );

		$admins_total   = is_array( $admin_ids ) ? count( $admin_ids ) : 0;
		$admins_enabled = 0;

		if ( $admins_total > 0 ) {
			foreach ( $admin_ids as $u ) {
				$v = get_user_meta( (int) $u->ID, 'aegisshield_2fa_enabled', true );
				// module stores various truthy forms; treat common “enabled” values as on.
				if ( $v === '1' || $v === 1 || $v === true || $v === 'on' || $v === 'yes' ) {
					$admins_enabled++;
				}
			}
		}

		$data = array(
			'totals' => array(
				'tracked_ips'     => $tracked_ips,
				'active_lockouts' => $active_lockouts,
				'recent_24h'      => $recent_24h,
			),
			'activity' => array(
				'labels' => $activity_labels,
				'values' => $activity_values,
			),
			'top_ips' => array(
				'labels' => $top_labels,
				'values' => $top_values,
			),
			'lockouts' => array(
				'active'     => $active_lockouts,
				'not_locked' => $not_locked,
			),
			'mfa' => array(
				'admins_total'   => $admins_total,
				'admins_enabled' => $admins_enabled,
			),
		);
	}

        protected function save_overview_tab() {
        check_admin_referer( 'aegisshield_login_guard_overview' );

        $settings = $this->plugin->get_settings();

        $max_attempts_raw = isset( $_POST['max_attempts'] ) ? absint( wp_unslash( $_POST['max_attempts'] ) ) : 0;
        $lock_minutes_raw = isset( $_POST['lockout_minutes'] ) ? absint( wp_unslash( $_POST['lockout_minutes'] ) ) : 0;

        $settings->set( 'login_guard', 'max_attempts', max( 1, $max_attempts_raw ) );
        $settings->set( 'login_guard', 'lockout_minutes', max( 1, $lock_minutes_raw ) );

        $settings->set( 'login_guard', 'block_unknown_user', isset( $_POST['block_unknown_user'] ) ? 1 : 0 );
        $settings->set( 'login_guard', 'honeypot_enabled', isset( $_POST['honeypot_enabled'] ) ? 1 : 0 );

        $settings->save();

        wp_safe_redirect( add_query_arg( array( 'tab' => 'overview', 'settings-updated' => '1' ) ) );
        exit;
    }

    /* ================================================================
     *  AUTHENTICATION TAB (MFA & TRUSTED DEVICES) — PRO ONLY
     * ================================================================ */

    protected function render_authentication_tab() {
        if ( ! $this->is_pro_active() ) {
            $this->render_pro_locked_notice();
            return;
        }

        $settings = $this->get_login_guard_settings();

        $enabled = ( isset( $settings['pro_2fa_enabled'] ) && 'on' === $settings['pro_2fa_enabled'] );

        $methods = is_array( $settings['pro_2fa_methods'] ?? null )
            ? $settings['pro_2fa_methods']
            : array( 'totp' );

        $enforced = is_array( $settings['pro_2fa_enforced_roles'] ?? null )
            ? $settings['pro_2fa_enforced_roles']
            : array( 'administrator', 'editor' );

        $trusted_enabled = ( isset( $settings['pro_trusted_devices_enabled'] ) && 'on' === $settings['pro_trusted_devices_enabled'] );

        $lifetime_days = isset( $settings['pro_trusted_device_lifetime_days'] )
            ? (int) $settings['pro_trusted_device_lifetime_days']
            : 30;

        $roles = wp_roles()->roles;

        ?>
        <form method="post">
            <?php wp_nonce_field( 'aegisshield_login_guard_auth' ); ?>
            <input type="hidden" name="aegisshield_login_guard_save_auth" value="1" />

            <h2><?php esc_html_e( 'Multi-Factor Authentication (MFA) & Trusted Devices', 'aegisshield-security' ); ?></h2>

            <table class="form-table">
                <tr>
                    <th><?php esc_html_e( 'Enable MFA', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="pro_2fa_enabled" value="1" <?php checked( $enabled ); ?> />
                        <?php esc_html_e( 'Require MFA (TOTP) for selected roles', 'aegisshield-security' ); ?></label>
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Allowed Methods', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="pro_2fa_methods[]" value="totp" <?php checked( in_array( 'totp', $methods, true ) ); ?> />
                            TOTP (Google Authenticator / Authy)
                        </label><br>
						<br />
						<label>
							<input type="checkbox"
								   name="pro_2fa_methods[]"
								   value="email"
								   <?php checked( in_array( 'email', $methods, true ) ); ?> />
							<?php esc_html_e( 'Email one-time code (Pro)', 'aegisshield-security' ); ?>
						</label>
                        <label><input type="checkbox" disabled /> Email (future)</label><br>
                        <label><input type="checkbox" disabled /> SMS (future)</label>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'QR Code Engine', 'aegisshield-security' ); ?></th>
                    <td>
                        <?php
                            $qr_engine = $settings['pro_mfa_qr_engine'] ?? 'local';
                        ?>
                        <select name="pro_mfa_qr_engine">
                            <option value="local" <?php selected( $qr_engine, 'local' ); ?>>
                                <?php esc_html_e( 'Local (AegisShield Recommended)', 'aegisshield-security' ); ?>
                            </option>
                            <option value="google" <?php selected( $qr_engine, 'google' ); ?>>
                                <?php esc_html_e( 'Google Charts API', 'aegisshield-security' ); ?>
                            </option>
                        </select>
                        <p class="description">
                            <?php esc_html_e( 'Local SVG works without external requests. Google Charts uses chart.googleapis.com to render the QR.', 'aegisshield-security' ); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Enforced Roles', 'aegisshield-security' ); ?></th>
                    <td>
                        <?php foreach ( $roles as $key => $role ) : ?>
                            <label><input type="checkbox" name="pro_2fa_enforced_roles[]" value="<?php echo esc_attr( $key ); ?>"
                                <?php checked( in_array( $key, $enforced, true ) ); ?> />
                                <?php echo esc_html( $role['name'] ); ?>
                            </label><br>
                        <?php endforeach; ?>
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Trusted Devices', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="pro_trusted_devices_enabled" value="1" <?php checked( $trusted_enabled ); ?> />
                            <?php esc_html_e( 'Enable trusted device recognition', 'aegisshield-security' ); ?>
                        </label>

                        <p>
                            <label><?php esc_html_e( 'Lifetime:', 'aegisshield-security' ); ?></label>
                            <select name="pro_trusted_device_lifetime_days">
                                <option value="30" <?php selected( $lifetime_days, 30 ); ?>>30 days</option>
                                <option value="60" <?php selected( $lifetime_days, 60 ); ?>>60 days</option>
                                <option value="90" <?php selected( $lifetime_days, 90 ); ?>>90 days</option>
                            </select>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button( 'Save Authentication Settings' ); ?>
        </form>
        <?php
    }

    protected function save_authentication_tab() {
        check_admin_referer( 'aegisshield_login_guard_auth' );

        $settings = $this->plugin->get_settings();
        $old_enabled = (string) $settings->get( 'login_guard', 'pro_2fa_enabled', 'off' );
        $old_methods = (array) $settings->get( 'login_guard', 'pro_2fa_methods', array( 'totp' ) );
        $old_qr      = (string) $settings->get( 'login_guard', 'pro_mfa_qr_engine', 'local' );
        $old_trusted = (string) $settings->get( 'login_guard', 'pro_trusted_devices_enabled', 'off' );
        $old_lifetime = (int) $settings->get( 'login_guard', 'pro_trusted_device_lifetime_days', 30 );



        $settings->set( 'login_guard', 'pro_2fa_enabled', isset( $_POST['pro_2fa_enabled'] ) ? 'on' : 'off' );

        $new_enabled = isset( $_POST['pro_2fa_enabled'] ) ? 'on' : 'off';
		if ( $new_enabled !== $old_enabled ) {
			$logger = $this->plugin->get_logger();
			if ( $logger ) {
				$logger->log(
					( 'on' === $new_enabled ? 'mfa_enabled' : 'mfa_disabled' ),
					( 'on' === $new_enabled ? __( 'MFA enabled.', 'aegisshield-security' ) : __( 'MFA disabled.', 'aegisshield-security' ) ),
					( 'on' === $new_enabled ? 'low' : 'critical' ),
					array(
						'old'      => $old_enabled,
						'new'      => $new_enabled,
						'user_id'  => get_current_user_id(),
					)
				);
			}
		}

        $methods = array();
        if ( isset( $_POST['pro_2fa_methods'] ) && is_array( $_POST['pro_2fa_methods'] ) ) {
            foreach ( (array) wp_unslash( $_POST['pro_2fa_methods'] ) as $m ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $methods[] = sanitize_key( $m );
            }
        }
        $settings->set( 'login_guard', 'pro_2fa_methods', $methods );

        $roles = array();
        if ( isset( $_POST['pro_2fa_enforced_roles'] ) && is_array( $_POST['pro_2fa_enforced_roles'] ) ) {
            foreach ( (array) wp_unslash( $_POST['pro_2fa_enforced_roles'] ) as $r ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $roles[] = sanitize_key( $r );
            }
        }
        $settings->set( 'login_guard', 'pro_2fa_enforced_roles', $roles );

        $qr_engine = sanitize_key( isset( $_POST['pro_mfa_qr_engine'] ) ? wp_unslash( $_POST['pro_mfa_qr_engine'] ) : 'local' );
        $settings->set( 'login_guard', 'pro_mfa_qr_engine', $qr_engine );

        $new_methods = $methods;
        $new_qr      = (string) $qr_engine;
		if ( $new_methods !== $old_methods || $new_qr !== $old_qr ) {
			$logger = $this->plugin->get_logger();
			if ( $logger ) {
				$logger->log(
					'mfa_policy_changed',
					__( 'MFA policy settings changed.', 'aegisshield-security' ),
					'medium',
					array(
						'old_methods' => $old_methods,
						'new_methods' => $new_methods,
						'old_qr'      => $old_qr,
						'new_qr'      => $new_qr,
						'user_id'     => get_current_user_id(),
					)
				);
			}
		}

        $settings->set( 'login_guard', 'pro_trusted_devices_enabled', isset( $_POST['pro_trusted_devices_enabled'] ) ? 'on' : 'off' );

        $new_trusted = isset( $_POST['pro_trusted_devices_enabled'] ) ? 'on' : 'off';
		if ( $new_trusted !== $old_trusted ) {
			$logger = $this->plugin->get_logger();
			if ( $logger ) {
				$logger->log(
					'trusted_devices_' . ( 'on' === $new_trusted ? 'enabled' : 'disabled' ),
					( 'on' === $new_trusted
						? __( 'Trusted devices enabled.', 'aegisshield-security' )
						: __( 'Trusted devices disabled.', 'aegisshield-security' )
					),
					( 'on' === $new_trusted ? 'low' : 'medium' ),
					array(
						'old'     => $old_trusted,
						'new'     => $new_trusted,
						'user_id' => get_current_user_id(),
					)
				);
			}
		}


        $lifetime = isset( $_POST['pro_trusted_device_lifetime_days'] )
            ? max( 1, absint( wp_unslash( $_POST['pro_trusted_device_lifetime_days'] ) ) )
            : 30;
        $settings->set( 'login_guard', 'pro_trusted_device_lifetime_days', $lifetime );

		if ( (int) $lifetime !== (int) $old_lifetime ) {
			$logger = $this->plugin->get_logger();
			if ( $logger ) {
				$logger->log(
					'trusted_devices_policy_changed',
					__( 'Trusted device lifetime changed.', 'aegisshield-security' ),
					'low',
					array(
						'old'     => (int) $old_lifetime,
						'new'     => (int) $lifetime,
						'user_id' => get_current_user_id(),
					)
				);
			}
		}


        $settings->save();

        wp_safe_redirect( add_query_arg( array( 'tab' => 'auth', 'settings-updated' => '1' ) ) );
        exit;
    }

    protected function render_lockouts_tab() {
        if ( ! $this->is_pro_active() ) {
            $this->render_pro_locked_notice();
            return;
        }

        global $wpdb;
        $settings = $this->get_login_guard_settings();

        $max_attempts = (int) ( $settings['max_attempts'] ?? 5 );
        $lock_minutes = (int) ( $settings['lockout_minutes'] ?? 15 );

        $t_known   = (int) ( $settings['pro_lockout_known_threshold']   ?? $max_attempts );
        $t_unknown = (int) ( $settings['pro_lockout_unknown_threshold'] ?? $max_attempts );
        $t_admin   = (int) ( $settings['pro_lockout_admin_threshold']   ?? $max_attempts );

        $progressive = ( 'on' === ( $settings['pro_lockout_progressive_enabled'] ?? 'off' ) );
        $enum_detect = ( 'on' === ( $settings['pro_username_enum_detection_enabled'] ?? 'off' ) );

        $table = $wpdb->prefix . 'aegisshield_login_attempts';
        $now   = current_time( 'mysql' );

        $locked = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table query; results cached via wp_cache_* above.
            $wpdb->prepare(
                "SELECT ip_address, attempts, last_attempt, locked_until
                 FROM %i
                 WHERE locked_until > %s
                 ORDER BY locked_until DESC LIMIT 200",
                $table,
                $now
            )
        );

        ?>
        <form method="post">
            <?php wp_nonce_field( 'aegisshield_login_guard_lockouts' ); ?>
            <input type="hidden" name="aegisshield_login_guard_save_lockouts" value="1" />

            <h2><?php esc_html_e( 'Lockouts & Rate Limits', 'aegisshield-security' ); ?></h2>

            <table class="form-table">
                <tr>
                    <th><?php esc_html_e( 'Default max attempts', 'aegisshield-security' ); ?></th>
                    <td><input type="number" name="max_attempts" class="small-text" min="1"
                        value="<?php echo esc_attr( $max_attempts ); ?>"></td>
                </tr>
                <tr>
                    <th><?php esc_html_e( 'Default lockout duration (minutes)', 'aegisshield-security' ); ?></th>
                    <td><input type="number" name="lockout_minutes" class="small-text" min="1"
                        value="<?php echo esc_attr( $lock_minutes ); ?>"></td>
                </tr>

                <tr><th><?php esc_html_e( 'Known users threshold (PRO)', 'aegisshield-security' ); ?></th>
                    <td><input type="number" name="pro_lockout_known_threshold" class="small-text"
                        value="<?php echo esc_attr( $t_known ); ?>"></td></tr>

                <tr><th><?php esc_html_e( 'Unknown username threshold (PRO)', 'aegisshield-security' ); ?></th>
                    <td><input type="number" name="pro_lockout_unknown_threshold" class="small-text"
                        value="<?php echo esc_attr( $t_unknown ); ?>"></td></tr>

                <tr><th><?php esc_html_e( 'Admin threshold (PRO)', 'aegisshield-security' ); ?></th>
                    <td><input type="number" name="pro_lockout_admin_threshold" class="small-text"
                        value="<?php echo esc_attr( $t_admin ); ?>"></td></tr>

                <tr>
                    <th><?php esc_html_e( 'Advanced Behavior', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="pro_lockout_progressive_enabled" value="1" <?php checked( $progressive ); ?> />
                            <?php esc_html_e( 'Progressive lockouts', 'aegisshield-security' ); ?></label>
                        <br>
                        <label><input type="checkbox" name="pro_username_enum_detection_enabled" value="1" <?php checked( $enum_detect ); ?> />
                            <?php esc_html_e( 'Username enumeration detection', 'aegisshield-security' ); ?></label>
                    </td>
                </tr>
            </table>

            <?php submit_button( __( 'Save Lockout Settings', 'aegisshield-security' ) ); ?>
        </form>

        <hr>

        <h2><?php esc_html_e( 'Locked IP Addresses', 'aegisshield-security' ); ?></h2>

        <?php if ( $locked ) : ?>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th>IP</th>
                        <th><?php esc_html_e( 'Attempts', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Last Attempt', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Locked Until', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ( $locked as $row ) : ?>
                        <tr>
                            <td><?php echo esc_html( $row->ip_address ); ?></td>
                            <td><?php echo esc_html( $row->attempts ); ?></td>
                            <td><?php echo esc_html( $row->last_attempt ); ?></td>
                            <td><?php echo esc_html( $row->locked_until ); ?></td>
                            <td>
                                <form method="post">
                                    <?php wp_nonce_field( 'aegisshield_unlock_ip' ); ?>
                                    <input type="hidden" name="ip_address" value="<?php echo esc_attr( $row->ip_address ); ?>">
                                    <input type="submit" name="aegisshield_unlock_ip" class="button secondary" value="Unlock">
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else : ?>
            <p><?php esc_html_e( 'No locked IPs.', 'aegisshield-security' ); ?></p>
        <?php endif; ?>
        <?php
    }

        protected function save_lockouts_tab() {
        // Only called if PRO is active (gated in render()).
        check_admin_referer( 'aegisshield_login_guard_lockouts' );

        $settings = $this->plugin->get_settings();

        $max_attempts_raw = isset( $_POST['max_attempts'] ) ? absint( wp_unslash( $_POST['max_attempts'] ) ) : 0;
        $lock_minutes_raw = isset( $_POST['lockout_minutes'] ) ? absint( wp_unslash( $_POST['lockout_minutes'] ) ) : 0;

        $t_known_raw   = isset( $_POST['pro_lockout_known_threshold'] ) ? absint( wp_unslash( $_POST['pro_lockout_known_threshold'] ) ) : 0;
        $t_unknown_raw = isset( $_POST['pro_lockout_unknown_threshold'] ) ? absint( wp_unslash( $_POST['pro_lockout_unknown_threshold'] ) ) : 0;
        $t_admin_raw   = isset( $_POST['pro_lockout_admin_threshold'] ) ? absint( wp_unslash( $_POST['pro_lockout_admin_threshold'] ) ) : 0;

        $settings->set( 'login_guard', 'max_attempts', max( 1, $max_attempts_raw ) );
        $settings->set( 'login_guard', 'lockout_minutes', max( 1, $lock_minutes_raw ) );

        $settings->set( 'login_guard', 'pro_lockout_known_threshold', max( 1, $t_known_raw ) );
        $settings->set( 'login_guard', 'pro_lockout_unknown_threshold', max( 1, $t_unknown_raw ) );
        $settings->set( 'login_guard', 'pro_lockout_admin_threshold', max( 1, $t_admin_raw ) );

        $settings->set( 'login_guard', 'pro_lockout_progressive_enabled', isset( $_POST['pro_lockout_progressive_enabled'] ) ? 'on' : 'off' );
        $settings->set( 'login_guard', 'pro_username_enum_detection_enabled', isset( $_POST['pro_username_enum_detection_enabled'] ) ? 'on' : 'off' );

        $settings->save();

        wp_safe_redirect( add_query_arg( array( 'tab' => 'lockouts', 'settings-updated' => '1' ) ) );
        exit;
    }

    protected function handle_unlock_ip() {
        global $wpdb;

        check_admin_referer( 'aegisshield_unlock_ip' ); $ip = sanitize_text_field( wp_unslash( $_POST['ip_address'] ?? '' ) );

        if ( ! $ip ) {
            return;
        }

        $table = $wpdb->prefix . 'aegisshield_login_attempts';

        $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            $table,
            array( 'locked_until' => null, 'attempts' => 0 ),
            array( 'ip_address' => $ip )
        );
    }

    protected function render_geo_tab() {
        if ( ! $this->is_pro_active() ) {
            $this->render_pro_locked_notice();
            return;
        }

        $settings = $this->get_login_guard_settings();

        $mode  = $settings['pro_geo_mode'] ?? 'alert';
        $codes = $settings['pro_allowed_countries'] ?? '';
        $hp    = ( 'on' === ( $settings['pro_honeypot_enabled'] ?? 'off' ) );
        $slug  = $settings['pro_honeypot_slug'] ?? '';

        ?>
        <form method="post">
            <?php wp_nonce_field( 'aegisshield_login_guard_geo' ); ?>
            <input type="hidden" name="aegisshield_login_guard_save_geo" value="1" />

            <h2><?php esc_html_e( 'Geo Rules & Honeypot', 'aegisshield-security' ); ?></h2>

            <table class="form-table">
                <tr>
                    <th><?php esc_html_e( 'Geo Rule Mode', 'aegisshield-security' ); ?></th>
                    <td>
                        <select name="pro_geo_mode">
                            <option value="off"   <?php selected( $mode, 'off' ); ?>>Off</option>
                            <option value="alert" <?php selected( $mode, 'alert' ); ?>>Alert Only</option>
                            <option value="block" <?php selected( $mode, 'block' ); ?>>Block Only</option>
                            <option value="both"  <?php selected( $mode, 'both' ); ?>>Alert + Block</option>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Allowed Countries', 'aegisshield-security' ); ?></th>
                    <td>
                        <textarea name="pro_allowed_countries" class="large-text" rows="3"><?php echo esc_textarea( $codes ); ?></textarea>
                        <p class="description"><?php esc_html_e( 'Comma-separated ISO codes (US, CA, GB). Leave empty to allow all.', 'aegisshield-security' ); ?></p>
                    </td>
                </tr>

                <tr>
                    <th><?php esc_html_e( 'Honeypot Login URL', 'aegisshield-security' ); ?></th>
                    <td>
                        <label><input type="checkbox" name="pro_honeypot_enabled" value="1" <?php checked( $hp ); ?> /> <?php esc_html_e( 'Enable', 'aegisshield-security' ); ?></label>
                        <br><br>
                        <label><?php esc_html_e( 'Slug:', 'aegisshield-security' ); ?></label>
                        <input type="text" name="pro_honeypot_slug" class="regular-text"
                            value="<?php echo esc_attr( $slug ); ?>" />
                    </td>
                </tr>
            </table>

            <?php submit_button( __( 'Save Geo Settings', 'aegisshield-security' ) ); ?>
        </form>
        <?php
    }

    protected function save_geo_tab() {
        check_admin_referer( 'aegisshield_login_guard_geo' );

        $settings = $this->plugin->get_settings();

        $settings->set(
            'login_guard',
            'pro_geo_mode',
            sanitize_key( isset( $_POST['pro_geo_mode'] ) ? wp_unslash( $_POST['pro_geo_mode'] ) : 'alert' )
        );

        $settings->set(
            'login_guard',
            'pro_allowed_countries',
            sanitize_textarea_field( isset( $_POST['pro_allowed_countries'] ) ? wp_unslash( $_POST['pro_allowed_countries'] ) : '' )
        );

        $settings->set(
            'login_guard',
            'pro_honeypot_enabled',
            isset( $_POST['pro_honeypot_enabled'] ) ? 'on' : 'off'
        );

        $slug = sanitize_title( isset( $_POST['pro_honeypot_slug'] ) ? wp_unslash( $_POST['pro_honeypot_slug'] ) : '' );
        $settings->set( 'login_guard', 'pro_honeypot_slug', $slug );

        $settings->save();

        wp_safe_redirect( add_query_arg( array( 'tab' => 'geo', 'settings-updated' => '1' ) ) );
        exit;
    }

}