<?php
namespace AegisShield\Admin_Pages;

use AegisShield\AS_Plugin;
use AegisShield\Modules\AS_Module_Malware;
use AegisShield\Modules\Malware\AS_Malware_Profiles;

defined( 'ABSPATH' ) || exit;

class AS_Page_Malware {

    protected $plugin;

	public function __construct( AS_Plugin $plugin ) {
		$this->plugin = $plugin;
	}

	protected function is_pro_active() {
		return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
	}

	protected function render_pro_locked_notice() {
		?>
		<div class="notice notice-warning inline">
			<p><strong><?php esc_html_e( 'This feature is available in AegisShield Pro.', 'aegisshield-security' ); ?></strong></p>
			<p><?php esc_html_e( 'Upgrade to unlock scheduled scans, advanced scan profiles, incremental quick scanning, and full Attack Story correlation.', 'aegisshield-security' ); ?></p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=aegisshield-license' ) ); ?>"
				   class="button button-primary">
				   <?php esc_html_e( 'Upgrade to AegisShield Pro', 'aegisshield-security' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

    public function render() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'aegisshield-security' ) );
        }

        $settings = $this->plugin->get_settings();
        $logger   = $this->plugin->get_logger();

		$section = 'malware';

		$normalize_rel_path = function ( $path ) {
			$path = (string) $path;
			$path = trim( $path );

			$path = str_replace( '\\', '/', $path );

			$abs = str_replace( '\\', '/', (string) ABSPATH );
			if ( $abs && 0 === strpos( $path, $abs ) ) {
				$path = substr( $path, strlen( $abs ) );
			}

			$path = ltrim( $path, '/' );

			return $path;
		};

		$is_pro = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();

        $handling_mode    = (string) $settings->get( $section, 'handling_mode', '' );
        $ignore_list      = $settings->get( $section, 'ignore_list', array() );
		$closed_list      = $settings->get( $section, 'closed_list', array() );
        $quarantine_dir   = trailingslashit( wp_upload_dir()['basedir'] ) . 'aegisshield-quarantine';
        $quarantine_url   = trailingslashit( wp_upload_dir()['baseurl'] ) . 'aegisshield-quarantine';
        $profiles         = new AS_Malware_Profiles( $settings );
        $available_profiles = $profiles->get_profiles();

		if ( ! is_array( $ignore_list ) ) {
			$ignore_list = array();
		}
		if ( ! is_array( $closed_list ) ) {
			$closed_list = array();
		}

		$ignore_list = array_values( array_unique( array_map( $normalize_rel_path, $ignore_list ) ) );
		$closed_list = array_values( array_unique( array_map( $normalize_rel_path, $closed_list ) ) );

        $updated    = false;
        $scan_ran   = false;
        $scan_results = $settings->get( $section, 'last_results', array() );
        if ( ! is_array( $scan_results ) ) {
            $scan_results = array();
        }

        if (
            isset( $_POST['aegisshield_malware_settings_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_malware_settings_nonce'] ) ),
                'aegisshield_malware_settings_save'
            )
        ) {
            $profile = isset( $_POST['malware_profile'] ) ? sanitize_text_field( wp_unslash( $_POST['malware_profile'] ) ) : 'balanced';

            if ( ! isset( $available_profiles[ $profile ] ) ) {
                $profile = 'balanced';
            }

            $directories = isset( $_POST['malware_dirs'] ) && is_array( $_POST['malware_dirs'] )
                ? array_map( 'sanitize_text_field', wp_unslash( $_POST['malware_dirs'] ) )
                : array();

            $custom_dirs = isset( $_POST['malware_custom_dirs'] ) && is_array( $_POST['malware_custom_dirs'] )
                ? array_map( 'sanitize_text_field', wp_unslash( $_POST['malware_custom_dirs'] ) )
                : array();

            $handling_mode_post = isset( $_POST['malware_handling_mode'] )
                ? sanitize_text_field( wp_unslash( $_POST['malware_handling_mode'] ) )
                : '';

            if ( ! in_array( $handling_mode_post, array( 'a', 'b', 'c' ), true ) ) {
                $handling_mode_post = 'a';
            }

            $settings->set( $section, 'profile', $profile );
            $settings->set( $section, 'directories', $directories );
            $settings->set( $section, 'custom_directories', $custom_dirs );
            $settings->set( $section, 'handling_mode', $handling_mode_post );
			$settings->set( $section, 'scan_dirs', $directories );
			$settings->set( $section, 'custom_dirs', $custom_dirs );

			if ( $logger ) {
				$logger->log(
					'malware_settings_engine_keys_written',
					array(
						'section'     => $section,
						'scan_dirs'   => $directories,
						'custom_dirs' => $custom_dirs,
						'user_id'     => get_current_user_id(),
					)
				);
			}
            $settings->save();

            $handling_mode = $handling_mode_post;

            $updated = true;

            if ( $logger ) {
                $logger->log(
                    'malware_settings_updated',
                    array(
                        'section'        => $section,
                        'profile'        => $profile,
                        'directories'    => $directories,
                        'custom_dirs'    => $custom_dirs,
                        'handling_mode'  => $handling_mode_post,
                        'user_id'        => get_current_user_id(),
                    )
                );
            }
        }

        if (
            isset( $_POST['aegisshield_malware_scan_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_malware_scan_nonce'] ) ),
                'aegisshield_malware_scan'
            )
            && isset( $_POST['run_malware_scan_now'] )
        ) {
            $module = null;

			if ( method_exists( $this->plugin, 'get_module' ) ) {
				$module = $this->plugin->get_module( 'malware' );

			} else {
				$module = new AS_Module_Malware( $this->plugin );
			}
            if ( $module instanceof AS_Module_Malware ) {

                $mode = isset( $_POST['malware_scan_mode'] ) ? sanitize_text_field( wp_unslash( $_POST['malware_scan_mode'] ) ) : 'heuristic';

                if ( 'core_checksums' === $mode && method_exists( $module, 'run_core_checksum_scan' ) ) {
                    $result = $module->run_core_checksum_scan();
                } else {
                    $result = $module->run_manual_scan();
                }

				if ( is_array( $result ) ) {
					$scan_results = ( isset( $result['results'] ) && is_array( $result['results'] ) )
						? $result['results']
						: $result;

					if ( $module instanceof AS_Module_Malware && ( ! isset( $result['results'] ) ) ) {
						$module->log_violation(
							'malware_manual_scan_return_shape',
							__( 'Manual scan returned results as a raw array (not wrapped). Page handled it via fallback.', 'aegisshield-security' ),
							array( 'count' => is_array( $scan_results ) ? count( $scan_results ) : 0 )
						);
					}
				} else {
					$scan_results = array();
					if ( $module instanceof AS_Module_Malware ) {
						$module->log_violation(
							'malware_manual_scan_failed',
							__( 'Manual scan did not return an array; scan may not have executed.', 'aegisshield-security' ),
							array()
						);
					}
				}

				$scan_ran = true;
            }
        }

        if (
            isset( $_POST['aegisshield_malware_action_nonce'] )
            && wp_verify_nonce(
                sanitize_text_field( wp_unslash( $_POST['aegisshield_malware_action_nonce'] ) ),
                'aegisshield_malware_action'
            )
            && isset( $_POST['file'], $_POST['malware_action'] )
        ) {
            $file_rel = sanitize_text_field( wp_unslash( $_POST['file'] ) );
			$file_rel = $normalize_rel_path( $file_rel );
            $action   = sanitize_text_field( wp_unslash( $_POST['malware_action'] ) );

            $from_wizard = ! empty( $_POST['malware_from_wizard'] );

            $profile_meta   = isset( $_POST['malware_profile_meta'] ) ? sanitize_text_field( wp_unslash( $_POST['malware_profile_meta'] ) ) : '';
            $incident_id    = isset( $_POST['malware_incident_id'] ) ? sanitize_text_field( wp_unslash( $_POST['malware_incident_id'] ) ) : '';
            $scan_type_meta = isset( $_POST['malware_scan_type'] ) ? sanitize_text_field( wp_unslash( $_POST['malware_scan_type'] ) ) : '';

			if ( $file_rel ) {

				if ( 'mark_safe' === $action ) {

					if ( in_array( $file_rel, $closed_list, true ) ) {
						$closed_list = array_values( array_diff( $closed_list, array( $file_rel ) ) );
						$settings->set( $section, 'closed_list', $closed_list );
					}

					if ( ! in_array( $file_rel, $ignore_list, true ) ) {
						$ignore_list[] = $file_rel;
						$settings->set( $section, 'ignore_list', $ignore_list );
					}

                    $allowlist_hashes = $settings->get( $section, 'allowlist_hashes', array() );
                    if ( ! is_array( $allowlist_hashes ) ) {
                        $allowlist_hashes = array();
                    }

                    $abs = trailingslashit( ABSPATH ) . ltrim( $file_rel, '/' );
                    if ( file_exists( $abs ) && is_file( $abs ) ) {
                        $sha = @hash_file( 'sha256', $abs );
                        if ( $sha ) {
                            $allowlist_hashes[ $file_rel ] = (string) $sha;
                            $settings->set( $section, 'allowlist_hashes', $allowlist_hashes );
                        }
                    }

					$settings->save();

					$meta = array(
						'file'        => $file_rel,
						'profile'     => $profile_meta,
						'incident_id' => $incident_id,
					);
					if ( $scan_type_meta ) {
						$meta['scan_type'] = $scan_type_meta;
					}
					if ( $from_wizard ) {
						$meta['from_wizard'] = true;
					}
					if ( $logger ) {
						$logger->log( 'malware_mark_safe', $meta );
					}

				} elseif ( 'mark_closed' === $action ) {

					if ( in_array( $file_rel, $ignore_list, true ) ) {
						$ignore_list = array_values( array_diff( $ignore_list, array( $file_rel ) ) );
						$settings->set( $section, 'ignore_list', $ignore_list );
					}

					if ( ! in_array( $file_rel, $closed_list, true ) ) {
						$closed_list[] = $file_rel;
						$settings->set( $section, 'closed_list', $closed_list );
					}

					$settings->save();

					$meta = array(
						'file'        => $file_rel,
						'profile'     => $profile_meta,
						'incident_id' => $incident_id,
					);
					if ( $scan_type_meta ) {
						$meta['scan_type'] = $scan_type_meta;
					}
					if ( $from_wizard ) {
						$meta['from_wizard'] = true;
					}
					if ( $logger ) {
						$logger->log( 'malware_mark_closed', $meta );
					}

				} elseif ( 'quarantine' === $action ) {

					if ( ! in_array( $handling_mode, array( 'b', 'c' ), true ) ) {
						if ( $module instanceof AS_Module_Malware ) {
							$module->log_violation(
								'malware_quarantine_blocked_mode',
								__( 'Quarantine requested but handling_mode does not allow it.', 'aegisshield-security' ),
								array( 'file' => $file_rel, 'handling_mode' => $handling_mode )
							);
						}
					} else {

						$ok = false;
						if ( $module instanceof AS_Module_Malware ) {
							$ok = $module->quarantine_rel( $file_rel );
						}

						if ( ! $ok && $module instanceof AS_Module_Malware ) {
							$module->log_violation(
								'malware_quarantine_failed',
								__( 'Quarantine attempted but did not complete successfully.', 'aegisshield-security' ),
								array( 'file' => $file_rel )
							);
						}

						if ( $ok ) {
							$meta = array(
								'file'        => $file_rel,
								'profile'     => $profile_meta,
								'incident_id' => $incident_id,
							);
							if ( $scan_type_meta ) { $meta['scan_type'] = $scan_type_meta; }
							if ( $from_wizard ) { $meta['from_wizard'] = true; }

							if ( $logger ) {
								$logger->log( 'malware_quarantined', $meta );
							}
						}
					}
				}
			}

			$redirect_args = array(
				'page' => 'aegisshield-malware',
				'tab'  => 'scanner',
			);

			if ( 'mark_safe' === $action ) {
				$redirect_args['as_ms_status'] = 'ignored';
			} elseif ( 'mark_closed' === $action ) {
				$redirect_args['as_ms_status'] = 'closed';
			}

			if ( isset( $_GET['as_ms_per_page'] ) ) {
				$redirect_args['as_ms_per_page'] = (int) $_GET['as_ms_per_page'];
			}

			if ( ! isset( $redirect_args['as_ms_status'] ) && isset( $_GET['as_ms_status'] ) ) {
				$redirect_args['as_ms_status'] = sanitize_key( wp_unslash( $_GET['as_ms_status'] ) );
			}

			if ( isset( $_GET['as_ms_sort'] ) ) {
				$redirect_args['as_ms_sort'] = sanitize_key( wp_unslash( $_GET['as_ms_sort'] ) );
			}
			if ( isset( $_GET['as_ms_dir'] ) ) {
				$redirect_args['as_ms_dir'] = strtolower( sanitize_text_field( wp_unslash( $_GET['as_ms_dir'] ) ) );
			}
			if ( isset( $_GET['as_ms_paged'] ) ) {
				$redirect_args['as_ms_paged'] = (int) $_GET['as_ms_paged'];
			}

			wp_safe_redirect( add_query_arg( $redirect_args, admin_url( 'admin.php' ) ) );
			exit;

        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'AegisShield Malware Scan', 'aegisshield-security' ); ?></h1>
			<p><?php esc_html_e( 'The AegisShield Malware Module provides proactive malware detection and response by scanning key WordPress directories for suspicious and potentially malicious code using heuristic analysis. It helps administrators identify risky files early, take controlled actions such as reviewing, ignoring, or quarantining threats, and maintain visibility through detailed logging and incident tracking. By combining flexible scan options, automation, and clear enforcement workflows, the Malware Module supports ongoing site integrity and reduces the risk of unnoticed compromise or reinfection.', 'aegisshield-security' ); ?></p>

            <?php if ( $updated ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Malware scan settings saved.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( $scan_ran ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e( 'Malware scan completed. Latest results are shown below.', 'aegisshield-security' ); ?></p>
                </div>
            <?php endif; ?>

            <?php
            $current_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'scanner';
            $page_url    = admin_url( 'admin.php?page=aegisshield-malware' );

            $review_file = '';
            if ( isset( $_GET['malware_review'] ) ) {
                $review_file = sanitize_text_field( wp_unslash( $_GET['malware_review'] ) );
            }
            ?>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'scanner', $page_url ) ); ?>" class="nav-tab <?php echo ( 'scanner' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Malware Scanner', 'aegisshield-security' ); ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'scheduler', $page_url ) ); ?>" class="nav-tab <?php echo ( 'scheduler' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Scheduled Scans', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'profiles', $page_url ) ); ?>" class="nav-tab <?php echo ( 'profiles' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Scan Profiles', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'incremental', $page_url ) ); ?>" class="nav-tab <?php echo ( 'incremental' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Incremental Quick Scan', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
                </a>
                <a href="<?php echo esc_url( add_query_arg( 'tab', 'attack_story', $page_url ) ); ?>" class="nav-tab <?php echo ( 'attack_story' === $current_tab ) ? 'nav-tab-active' : ''; ?>">
                    <?php esc_html_e( 'Attack Story', 'aegisshield-security' ); ?>
					<?php if ( ! $is_pro ) : ?>
					<span class="aegisshield-badge-pro" style="margin-left:4px;font-size:11px;padding:2px 5px;border-radius:3px;background:#ff9800;color:#fff;text-transform:uppercase;">
						<?php esc_html_e( 'Pro', 'aegisshield-security' ); ?>
					</span>
				<?php endif; ?>
                </a>
            </h2>

            <div class="aegisshield-tab-panel">
                <?php if ( 'scanner' === $current_tab ) : ?>
				    <?php $this->render_malware_dashboard_visuals(); ?>

                    <h2><?php esc_html_e( 'Scan Settings', 'aegisshield-security' ); ?></h2>

                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_malware_settings_save', 'aegisshield_malware_settings_nonce' ); ?>

                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="malware_profile"><?php esc_html_e( 'Scan profile', 'aegisshield-security' ); ?></label>
                                </th>
                                <td>
                                    <select id="malware_profile" name="malware_profile">
                                        <?php foreach ( $available_profiles as $profile_id => $profile_data ) : ?>
                                            <option value="<?php echo esc_attr( $profile_id ); ?>" <?php selected( $settings->get( $section, 'profile', 'balanced' ), $profile_id ); ?>>
                                                <?php echo esc_html( $profile_data['label'] ); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="description">
                                        <?php esc_html_e( 'Choose how aggressive the malware heuristics should be.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php esc_html_e( 'Directories to scan', 'aegisshield-security' ); ?></th>
                                <td>
                                    <?php
                                    $selected_dirs = $settings->get( $section, 'directories', array( 'wp-content/plugins', 'wp-content/themes' ) );
                                    if ( ! is_array( $selected_dirs ) ) {
                                        $selected_dirs = array();
                                    }

                                    $builtin_dirs = array(
                                        'wp-content/plugins',
                                        'wp-content/themes',
                                        'wp-content/uploads',
                                        'wp-admin',
                                    );
                                    ?>
                                    <fieldset>
                                        <?php foreach ( $builtin_dirs as $dir ) : ?>
                                            <label>
                                                <input type="checkbox" name="malware_dirs[]" value="<?php echo esc_attr( $dir ); ?>" <?php checked( in_array( $dir, $selected_dirs, true ) ); ?> />
                                                <code><?php echo esc_html( $dir ); ?></code>
                                            </label>
                                            <br />
                                        <?php endforeach; ?>
                                    </fieldset>
                                    <p class="description">
                                        <?php esc_html_e( 'These core paths will be scanned recursively for suspicious PHP files, themes, and plugins.', 'aegisshield-security' ); ?>
                                    </p>

                                    <h4><?php esc_html_e( 'Custom directories (Pro)', 'aegisshield-security' ); ?></h4>
                                    <?php
                                    $custom_dirs = $settings->get( $section, 'custom_directories', array() );
                                    if ( ! is_array( $custom_dirs ) ) {
                                        $custom_dirs = array();
                                    }
                                    ?>
                                    <p class="description">
                                        <?php esc_html_e( 'Add additional paths for AegisShield to include in malware scans (e.g., mu-plugins, vendor libraries).', 'aegisshield-security' ); ?>
                                    </p>
                                    <div id="aegisshield-malware-custom-dirs">
                                        <?php if ( empty( $custom_dirs ) ) : ?>
                                            <p class="description">
                                                <?php esc_html_e( 'No custom paths defined yet.', 'aegisshield-security' ); ?>
                                            </p>
                                        <?php else : ?>
                                            <ul>
                                                <?php foreach ( $custom_dirs as $dir ) : ?>
                                                    <li>
                                                        <input type="text" name="malware_custom_dirs[]" value="<?php echo esc_attr( $dir ); ?>" class="regular-text" />
                                                    </li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php endif; ?>
                                    </div>
                                    <p>
                                        <button type="button" class="button" onclick="var c = document.getElementById('aegisshield-malware-custom-dirs'); var li = document.createElement('li'); li.innerHTML = '<input type=&quot;text&quot; name=&quot;malware_custom_dirs[]&quot; value=&quot;&quot; class=&quot;regular-text&quot; />'; if ( ! c.querySelector('ul') ) { var ul = document.createElement('ul'); c.innerHTML=''; c.appendChild(ul); } c.querySelector('ul').appendChild(li);">
                                            <?php esc_html_e( 'Add custom directory', 'aegisshield-security' ); ?>
                                        </button>
                                    </p>
                                    <?php if ( ! $is_pro ) : ?>
                                        <p class="description">
                                            <em><?php esc_html_e( 'Custom directory scanning is available in AegisShield Pro. Activate your license on the License & Upgrades page.', 'aegisshield-security' ); ?></em>
                                        </p>
                                    <?php endif; ?>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php esc_html_e( 'After detection', 'aegisshield-security' ); ?></th>
                                <td>
                                    <fieldset>
                                        <label>
                                            <input type="radio" name="malware_handling_mode" value="a" <?php checked( 'a', $handling_mode ); ?> />
                                            <?php esc_html_e( 'Log only (no automatic changes)', 'aegisshield-security' ); ?>
                                        </label>
                                        <br />
                                        <label>
                                            <input type="radio" name="malware_handling_mode" value="b" <?php checked( 'b', $handling_mode ); ?> />
                                            <?php esc_html_e( 'Quarantine high-risk files (recommended for most sites)', 'aegisshield-security' ); ?>
                                        </label>
                                        <br />
                                        <label>
                                            <input type="radio" name="malware_handling_mode" value="c" <?php checked( 'c', $handling_mode ); ?> />
                                            <?php esc_html_e( 'Quarantine medium and high-risk files (most aggressive)', 'aegisshield-security' ); ?>
                                        </label>
                                    </fieldset>
                                    <p class="description">
                                        <?php esc_html_e( 'Quarantined files are moved into a secure folder under wp-content/aegisshield-quarantine and replaced with a harmless placeholder.', 'aegisshield-security' ); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>

                        <?php submit_button( __( 'Save Changes', 'aegisshield-security' ) ); ?>
                    </form>

                    <h2><?php esc_html_e( 'Run Malware Scan', 'aegisshield-security' ); ?></h2>

                    <form method="post">
                        <?php wp_nonce_field( 'aegisshield_malware_scan', 'aegisshield_malware_scan_nonce' ); ?>
						<p>
                            <label for="malware_scan_mode"><strong><?php esc_html_e( 'Scan mode:', 'aegisshield-security' ); ?></strong></label>
                            <select name="malware_scan_mode" id="malware_scan_mode">
                                <option value="heuristic"><?php esc_html_e( 'Heuristic (Malware Rules v1)', 'aegisshield-security' ); ?></option>
                                <option value="core_checksums"><?php esc_html_e( 'Core Integrity (WordPress checksums)', 'aegisshield-security' ); ?></option>
                            </select>
                        </p>
                        <p>
                            <input type="submit" name="run_malware_scan_now" class="button button-primary" value="<?php esc_attr_e( 'Run Malware Scan Now', 'aegisshield-security' ); ?>" />
                        </p>
                        <p class="description">
                            <?php esc_html_e( 'Runs a heuristic scan on the selected directories using the current settings.', 'aegisshield-security' ); ?>
                        </p>
                    </form>

            <h2><?php esc_html_e( 'Latest Scan Results', 'aegisshield-security' ); ?></h2>
            <?php
            if ( ! empty( $scan_results ) ) {
                $allowed_per_page = array( 25, 50, 100, 200 );
                $per_page         = isset( $_GET['as_ms_per_page'] ) ? (int) $_GET['as_ms_per_page'] : 25;
                if ( ! in_array( $per_page, $allowed_per_page, true ) ) {
                    $per_page = 25;
                }

                $sort = isset( $_GET['as_ms_sort'] ) ? sanitize_key( wp_unslash( $_GET['as_ms_sort'] ) ) : 'score';
                $dir  = isset( $_GET['as_ms_dir'] ) ? strtolower( sanitize_text_field( wp_unslash( $_GET['as_ms_dir'] ) ) ) : 'desc';
                if ( 'asc' !== $dir ) {
                    $dir = 'desc';
                }

                $current_page = isset( $_GET['as_ms_paged'] ) ? (int) $_GET['as_ms_paged'] : 1;
                if ( $current_page < 1 ) {
                    $current_page = 1;
                }

				$status = isset( $_GET['as_ms_status'] )
					? sanitize_key( wp_unslash( $_GET['as_ms_status'] ) )
					: 'active';

				if ( ! in_array( $status, array( 'all', 'active', 'ignored', 'closed' ), true ) ) {
					$status = 'active';
				}


                $sorted_results = is_array( $scan_results ) ? array_values( $scan_results ) : array();

				foreach ( $sorted_results as &$entry ) {
					$file = isset( $entry['file'] ) ? $normalize_rel_path( $entry['file'] ) : '';

					if ( isset( $entry['file'] ) ) {
						$entry['file'] = $file;
					}

					if ( $file && in_array( $file, $ignore_list, true ) ) {
						$entry['status'] = 'ignored';
					} elseif ( $file && in_array( $file, $closed_list, true ) ) {
						$entry['status'] = 'closed';
					} else {
						$entry['status'] = 'active';
					}
				}
				unset( $entry );
				
				$sorted_results = array_values(
					array_filter(
						$sorted_results,
						function ( $entry ) use ( $status, $ignore_list, $closed_list ) {
							$file = isset( $entry['file'] ) ? (string) $entry['file'] : '';
							$is_ignored = ( '' !== $file ) && in_array( $file, $ignore_list, true );
							$is_closed  = ( '' !== $file ) && in_array( $file, $closed_list, true );

							if ( 'all' === $status ) {
								return true;
							}

							if ( 'ignored' === $status ) {
								return $is_ignored;
							}

							if ( 'closed' === $status ) {
								return $is_closed;
							}

							return ( ! $is_ignored && ! $is_closed );
						}
					)
				);

                usort(
                    $sorted_results,
                    function( $a, $b ) use ( $sort, $dir ) {
                        $direction = ( 'asc' === $dir ) ? 1 : -1;

                        $get_level_value = function( $level ) {
                            $map   = array(
                                'low'    => 1,
                                'medium' => 2,
                                'high'   => 3,
                            );
                            $level = strtolower( (string) $level );
                            return isset( $map[ $level ] ) ? $map[ $level ] : 0;
                        };

                        if ( 'risk' === $sort ) {
                            $a_val = $get_level_value( isset( $a['level'] ) ? $a['level'] : '' );
                            $b_val = $get_level_value( isset( $b['level'] ) ? $b['level'] : '' );
                        } elseif ( 'reason' === $sort ) {
                            $a_reason = '';
                            if ( isset( $a['reasons'] ) && is_array( $a['reasons'] ) && ! empty( $a['reasons'] ) ) {
                                $first    = reset( $a['reasons'] );
                                $a_reason = strtolower( (string) $first );
                            }

                            $b_reason = '';
                            if ( isset( $b['reasons'] ) && is_array( $b['reasons'] ) && ! empty( $b['reasons'] ) ) {
                                $first    = reset( $b['reasons'] );
                                $b_reason = strtolower( (string) $first );
                            }

                            $cmp = strcmp( $a_reason, $b_reason );
                            if ( 0 !== $cmp ) {
                                return $direction * $cmp;
                            }

                            // Fallback to file name.
                            $a_file = isset( $a['file'] ) ? (string) $a['file'] : '';
                            $b_file = isset( $b['file'] ) ? (string) $b['file'] : '';
                            return $direction * strcmp( $a_file, $b_file );
                        } elseif ( 'modified' === $sort ) {
                            $a_val = isset( $a['modified'] ) ? (int) $a['modified'] : 0;
                            $b_val = isset( $b['modified'] ) ? (int) $b['modified'] : 0;
                        } else { // score (default).
                            $a_val = isset( $a['score'] ) ? (int) $a['score'] : 0;
                            $b_val = isset( $b['score'] ) ? (int) $b['score'] : 0;
                        }

                        if ( $a_val === $b_val ) {
                            $a_file = isset( $a['file'] ) ? (string) $a['file'] : '';
                            $b_file = isset( $b['file'] ) ? (string) $b['file'] : '';
                            return $direction * strcmp( $a_file, $b_file );
                        }

                        return ( $a_val < $b_val ) ? -1 * $direction : 1 * $direction;
                    }
                );

                $total_items = count( $sorted_results );
                $total_pages = $total_items > 0 ? (int) max( 1, ceil( $total_items / $per_page ) ) : 1;

                if ( $current_page > $total_pages ) {
                    $current_page = $total_pages;
                }

                $offset        = ( $current_page - 1 ) * $per_page;
                $paged_results = array_slice( $sorted_results, $offset, $per_page );
            }
            ?>

            <?php if ( empty( $scan_results ) ) : ?>
                <p><?php esc_html_e( 'No suspicious files detected yet, or scans have not been run.', 'aegisshield-security' ); ?></p>
            <?php else : ?>

                <form id="as-malware-filters" method="get" style="margin-bottom:10px;">
                    <input type="hidden" name="page" value="aegisshield-malware" />
                    <input type="hidden" name="tab" value="scanner" />

                    <label>
                        <?php esc_html_e( 'Show:', 'aegisshield-security' ); ?>
                        <select name="as_ms_per_page">
                            <?php foreach ( $allowed_per_page as $size ) : ?>
                                <option value="<?php echo esc_attr( $size ); ?>" <?php selected( $per_page, $size ); ?>>
                                    <?php echo esc_html( $size ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>

                    <span style="margin-left:15px;"></span>

					<label>
						<?php esc_html_e( 'Status:', 'aegisshield-security' ); ?>
						<select name="as_ms_status">
							<option value="all" <?php selected( $status, 'all' ); ?>>
								<?php esc_html_e( 'All', 'aegisshield-security' ); ?>
							</option>
							<option value="active" <?php selected( $status, 'active' ); ?>>
								<?php esc_html_e( 'Active', 'aegisshield-security' ); ?>
							</option>
							<option value="closed" <?php selected( $status, 'closed' ); ?>>
								<?php esc_html_e( 'Closed', 'aegisshield-security' ); ?>
							</option>
							<option value="ignored" <?php selected( $status, 'ignored' ); ?>>
								<?php esc_html_e( 'Ignored', 'aegisshield-security' ); ?>
							</option>
						</select>
					</label>

                    <label>
                        <?php esc_html_e( 'Sort by:', 'aegisshield-security' ); ?>
                        <select name="as_ms_sort">
                            <option value="score" <?php selected( $sort, 'score' ); ?>>
                                <?php esc_html_e( 'Score', 'aegisshield-security' ); ?>
                            </option>
                            <option value="risk" <?php selected( $sort, 'risk' ); ?>>
                                <?php esc_html_e( 'Risk Level', 'aegisshield-security' ); ?>
                            </option>
                            <option value="reason" <?php selected( $sort, 'reason' ); ?>>
                                <?php esc_html_e( 'Reasons', 'aegisshield-security' ); ?>
                            </option>
                            <option value="modified" <?php selected( $sort, 'modified' ); ?>>
                                <?php esc_html_e( 'Last Modified', 'aegisshield-security' ); ?>
                            </option>
                        </select>
                    </label>

                    <span style="margin-left:15px;"></span>

                    <label>
                        <?php esc_html_e( 'Order:', 'aegisshield-security' ); ?>
                        <select name="as_ms_dir">
                            <option value="asc" <?php selected( $dir, 'asc' ); ?>>
                                <?php esc_html_e( 'Ascending', 'aegisshield-security' ); ?>
                            </option>
                            <option value="desc" <?php selected( $dir, 'desc' ); ?>>
                                <?php esc_html_e( 'Descending', 'aegisshield-security' ); ?>
                            </option>
                        </select>
                    </label>

                    <button type="submit" class="button" style="margin-left:10px;">
                        <?php esc_html_e( 'Apply', 'aegisshield-security' ); ?>
                    </button>
                </form>

                    <table class="widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'File', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Score', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Risk Level', 'aegisshield-security' ); ?></th>
								<th><?php esc_html_e( 'Status', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Reasons', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Last Modified', 'aegisshield-security' ); ?></th>
                                <th><?php esc_html_e( 'Actions', 'aegisshield-security' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ( $paged_results as $entry ) : ?>
                                <?php
                                $file      = isset( $entry['file'] ) ? (string) $entry['file'] : '';
                                $score     = isset( $entry['score'] ) ? (int) $entry['score'] : 0;
                                $level     = isset( $entry['level'] ) ? (string) $entry['level'] : '';
                                $reasons   = isset( $entry['reasons'] ) && is_array( $entry['reasons'] ) ? $entry['reasons'] : array();
                                $mtime     = isset( $entry['modified'] ) ? (int) $entry['modified'] : 0;
                                $mtime_str = '';

                                if ( $mtime > 0 ) {
                                    $mtime_str = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $mtime );
                                }

                                if ( 'high' === $level ) {
                                    $level_label = __( 'High', 'aegisshield-security' );
                                } elseif ( 'medium' === $level ) {
                                    $level_label = __( 'Medium', 'aegisshield-security' );
                                } else {
                                    $level_label = __( 'Low', 'aegisshield-security' );
                                }

                                $is_reviewed = ( '' !== $review_file && $review_file === $file );
                                ?>
                                <tr>
									<td><code><?php echo esc_html( $file ); ?></code></td>
									<td><?php echo esc_html( $score ); ?></td>
									<td><?php echo esc_html( $level_label ); ?></td>

									<?php
									$entry_status = isset( $entry['status'] ) ? (string) $entry['status'] : 'active';
									?>
									<td>
										<span class="as-status as-status-<?php echo esc_attr( $entry_status ); ?>">
											<?php echo esc_html( ucfirst( $entry_status ) ); ?>
										</span>
									</td>

									<td>
										<?php if ( ! empty( $reasons ) ) : ?>
											<ul style="margin:0;padding-left:18px;">
												<?php foreach ( $reasons as $reason ) : ?>
													<li><?php echo esc_html( $reason ); ?></li>
												<?php endforeach; ?>
											</ul>
										<?php endif; ?>
									</td>

									<td><?php echo esc_html( $mtime_str ); ?></td>

									<td>
										<form method="post" style="display:inline;">
											<?php wp_nonce_field( 'aegisshield_malware_action', 'aegisshield_malware_action_nonce' ); ?>
											<input type="hidden" name="file" value="<?php echo esc_attr( $file ); ?>" />

											<?php if ( $is_pro && $is_reviewed ) : ?>
												<input type="hidden" name="malware_from_wizard" value="1" />
											<?php endif; ?>

											<select name="malware_action">
												<option value=""><?php esc_html_e( 'Select action', 'aegisshield-security' ); ?></option>
												<option value="mark_safe"><?php esc_html_e( 'Mark as safe (ignore)', 'aegisshield-security' ); ?></option>
												<option value="mark_closed"><?php esc_html_e( 'Closed', 'aegisshield-security' ); ?></option>
												<?php if ( in_array( $handling_mode, array( 'b', 'c' ), true ) ) : ?>
													<option value="quarantine"><?php esc_html_e( 'Quarantine file', 'aegisshield-security' ); ?></option>
												<?php endif; ?>
											</select>

											<button type="submit" class="button button-secondary">
												<?php esc_html_e( 'Apply', 'aegisshield-security' ); ?>
											</button>
										</form>

										<?php if ( $is_pro ) : ?>
											<br />
											<a href="<?php echo esc_url( add_query_arg( 'malware_review', rawurlencode( $file ), $page_url ) ); ?>">
												<?php esc_html_e( 'Review & Cleanup (Pro)', 'aegisshield-security' ); ?>
											</a>
											<br />
											<a href="<?php echo esc_url( add_query_arg( 'file', rawurlencode( $file ), admin_url( 'admin.php?page=aegisshield-incident-report' ) ) ); ?>" target="_blank">
												<?php esc_html_e( 'Export Incident Report (HTML/PDF)', 'aegisshield-security' ); ?>
											</a>
										<?php endif; ?>
									</td>

								</tr>

                                <?php if ( $is_pro && $is_reviewed ) : ?>
                                    <tr class="as-ms-review-row">
                                        <td colspan="7">
                                            <div class="as-ms-review-panel" style="margin:10px 0;padding:12px;border:1px solid #ccd0d4;background:#f8f9fa;">
                                                <h3 style="margin-top:0;"><?php esc_html_e( 'Guided cleanup', 'aegisshield-security' ); ?></h3>
                                                <p>
                                                    <strong><?php esc_html_e( 'File:', 'aegisshield-security' ); ?></strong>
                                                    <code><?php echo esc_html( $file ); ?></code>
                                                </p>
                                                <p>
                                                    <strong><?php esc_html_e( 'Score:', 'aegisshield-security' ); ?></strong>
                                                    <?php echo esc_html( $score ); ?>
                                                    &nbsp;&nbsp;
                                                    <strong><?php esc_html_e( 'Risk level:', 'aegisshield-security' ); ?></strong>
                                                    <?php echo esc_html( $level_label ); ?>
                                                </p>
                                                <p class="description">
                                                    <?php esc_html_e( 'File Integrity diff and baseline restore will be available in a future update. For now, you can mark this file as safe or quarantine it. Both actions are fully logged for audit purposes.', 'aegisshield-security' ); ?>
                                                </p>

                                                <form method="post" style="margin-top:10px;">
                                                    <?php wp_nonce_field( 'aegisshield_malware_action', 'aegisshield_malware_action_nonce' ); ?>
                                                    <input type="hidden" name="file" value="<?php echo esc_attr( $file ); ?>" />
                                                    <input type="hidden" name="malware_from_wizard" value="1" />
                                                    <p>
                                                        <button type="submit" name="malware_action" value="mark_safe" class="button button-secondary">
                                                            <?php esc_html_e( 'Mark as safe (ignore)', 'aegisshield-security' ); ?>
                                                        </button>
                                                        <?php if ( in_array( $handling_mode, array( 'b', 'c' ), true ) ) : ?>
                                                            <button type="submit" name="malware_action" value="quarantine" class="button button-secondary">
                                                                <?php esc_html_e( 'Quarantine file', 'aegisshield-security' ); ?>
                                                            </button>
                                                        <?php endif; ?>
                                                    </p>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>

                    <?php
                    if ( $total_pages > 1 ) :
                        ?>
                        <div class="tablenav">
                            <div class="tablenav-pages">
                                <?php
                                printf(
                                    '<span class="displaying-num">' . esc_html(
                                        /* translators: %s: number of items. */
                                        _n(
                                            '%s item',
                                            '%s items',
                                            $total_items,
                                            'aegisshield-security'
                                        )
                                    ) . '</span>',
                                    esc_html( number_format_i18n( $total_items ) )
                                );
                                ?>
                                <span class="pagination-links">
                                    <?php
                                    $first_page = 1;
                                    $last_page  = $total_pages;
                                    $prev_page  = max( $first_page, $current_page - 1 );
                                    $next_page  = min( $last_page, $current_page + 1 );

									$base_args = array(
										'tab'            => 'scanner',
										'as_ms_per_page' => $per_page,
										'as_ms_status'   => $status,
										'as_ms_sort'     => $sort,
										'as_ms_dir'      => $dir,
									);
                                    ?>
                                    <a class="first-page<?php echo esc_attr( ( $current_page === $first_page ) ? ' disabled' : '' ); ?>"
                                       href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_ms_paged' => $first_page ) ), $page_url ) ); ?>">
                                        <span class="screen-reader-text"><?php esc_html_e( 'First page', 'aegisshield-security' ); ?></span>
                                        <span aria-hidden="true">&laquo;</span>
                                    </a>

                                    <a class="prev-page<?php echo ( $current_page === $first_page ) ? ' disabled' : ''; ?>"
                                       href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_ms_paged' => $prev_page ) ), $page_url ) ); ?>">
                                        <span class="screen-reader-text"><?php esc_html_e( 'Previous page', 'aegisshield-security' ); ?></span>
                                        <span aria-hidden="true">&lsaquo;</span>
                                    </a>

                                    <span class="paging-input">
                                        <?php
                                        printf(
                                            /* translators: 1: current page, 2: total pages. */
                                            esc_html__( '%1$s of %2$s', 'aegisshield-security' ),
                                            '<span class="current-page">' . esc_html( $current_page ) . '</span>',
                                            '<span class="total-pages">' . esc_html( $total_pages ) . '</span>'
                                        );
                                        ?>
                                    </span>

                                    <a class="next-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                       href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_ms_paged' => $next_page ) ), $page_url ) ); ?>">
                                        <span class="screen-reader-text"><?php esc_html_e( 'Next page', 'aegisshield-security' ); ?></span>
                                        <span aria-hidden="true">&rsaquo;</span>
                                    </a>

                                    <a class="last-page<?php echo ( $current_page === $last_page ) ? ' disabled' : ''; ?>"
                                       href="<?php echo esc_url( add_query_arg( array_merge( $base_args, array( 'as_ms_paged' => $last_page ) ), $page_url ) ); ?>">
                                        <span class="screen-reader-text"><?php esc_html_e( 'Last page', 'aegisshield-security' ); ?></span>
                                        <span aria-hidden="true">&raquo;</span>
                                    </a>
                                </span>
                            </div>
                        </div>
                    <?php endif; ?>
            <?php endif; ?>

				<?php elseif ( 'scheduler' === $current_tab ) : ?>

					<?php if ( ! $is_pro ) : ?>
					<?php $this->render_pro_locked_notice(); ?>
					<?php else : ?>
					<?php
					if ( class_exists( '\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Scheduler' ) ) {
						\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Scheduler::render_tab( $this->plugin );
					} else {
						echo '<p>' . esc_html__( 'Scheduler UI will be available in a future release.', 'aegisshield-security' ) . '</p>';
					}
					?>
					<?php endif; ?>
					
				<?php elseif ( 'profiles' === $current_tab ) : ?>

					<?php if ( ! $is_pro ) : ?>

						<?php $this->render_pro_locked_notice(); ?>

					<?php else : ?>

					<?php
					if ( class_exists( '\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Profiles' ) ) {
						\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Profiles::render_tab( $this->plugin );
					} else {
						echo '<p>' . esc_html__( 'Profile manager will be available in a future release.', 'aegisshield-security' ) . '</p>';
					}
					?>
					<?php endif; ?>

				<?php elseif ( 'incremental' === $current_tab ) : ?>

					<?php if ( ! $is_pro ) : ?>

						<?php $this->render_pro_locked_notice(); ?>

					<?php else : ?>

					<?php
					if ( class_exists( '\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Incremental' ) ) {
						\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Incremental::render_tab( $this->plugin );
					} else {
						echo '<p>' . esc_html__( 'In a future release, this tab will show incremental quick scans that only rescan changed files between baselines.', 'aegisshield-security' ) . '</p>';
					}
					?>

					<?php endif; ?>

				<?php elseif ( 'attack_story' === $current_tab ) : ?>

					<?php if ( ! $is_pro ) : ?>

						<?php $this->render_pro_locked_notice(); ?>

					<?php else : ?>

					<?php
					if ( class_exists( '\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Attack_Story' ) ) {
						\AegisShield\Admin_Pages\Malware\AS_Page_Malware_Tab_Attack_Story::render_tab( $this->plugin );
					} else {
						echo '<p>' . esc_html__( 'Attack Story view will be available in a future release.', 'aegisshield-security' ) . '</p>';
					}
					?>

                <?php endif; ?>
				
            <?php endif; ?>
				
            </div>
        </div>
        <?php
    }

	protected function render_malware_dashboard_visuals() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$data = $this->get_malware_dashboard_data();

		// Use local Chart.js (no CDN).
		$plugin_file = defined( 'AEGISSHIELD_PLUGIN_FILE' )
			? AEGISSHIELD_PLUGIN_FILE
			: dirname( __FILE__, 3 ) . '/aegisshield-security.php';

		$chart_src = plugins_url( 'assets/js/chart.umd.min.js', $plugin_file );
		?>
		<style>
			.aegis-mal-dash{
				display:grid;
				grid-template-columns:repeat(4,minmax(0,1fr));
				gap:12px;
				margin:12px 0 14px;
			}
			.aegis-mal-card{
				background:#fff;
				border:1px solid #dcdcde;
				border-radius:10px;
				padding:12px 12px 10px;
				box-shadow:0 1px 1px rgba(0,0,0,.04);
				min-height:235px;
			}
			.aegis-mal-card h3{
				margin:0 0 8px;
				font-size:13px;
				line-height:1.3;
				display:flex;
				align-items:center;
				gap:8px;
			}
			.aegis-mal-pill{
				font-size:12px;
				padding:2px 8px;
				border-radius:999px;
				border:1px solid #dcdcde;
				background:#f6f7f7;
			}
			.aegis-mal-canvas{ position:relative; height:175px; }
			.aegis-mal-sub{
				margin-top:8px;
				font-size:12px;
				color:#646970;
				display:flex;
				justify-content:space-between;
				gap:10px;
			}
			@media (max-width:1200px){ .aegis-mal-dash{ grid-template-columns:repeat(2,minmax(0,1fr)); } }
			@media (max-width:782px){ .aegis-mal-dash{ grid-template-columns:1fr; } }
		</style>

		<div class="aegis-mal-dash" id="aegis-mal-dash"
			 data-mal="<?php echo esc_attr( wp_json_encode( $data ) ); ?>">

			<div class="aegis-mal-card">
				<h3>
					<span class="dashicons dashicons-visibility"></span>
					<?php esc_html_e( 'Scan Profile Coverage', 'aegisshield-security' ); ?>
					<span class="aegis-mal-pill"><?php echo esc_html( (string) $data['profile']['enabled_count'] ); ?>/<?php echo esc_html( (string) count( $data['profile']['labels'] ) ); ?></span>
				</h3>
				<div class="aegis-mal-canvas"><canvas id="aegisMalChartProfile"></canvas></div>
				<div class="aegis-mal-sub">
					<span><?php esc_html_e( 'Targets enabled', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['profile']['mode_label'] ); ?></span>
				</div>
			</div>

			<div class="aegis-mal-card">
				<h3>
					<span class="dashicons dashicons-shield-alt"></span>
					<?php esc_html_e( 'Latest Scan: Suspect Ratio', 'aegisshield-security' ); ?>
					<span class="aegis-mal-pill"><?php echo esc_html( (string) $data['latest']['suspects'] ); ?></span>
				</h3>
				<div class="aegis-mal-canvas"><canvas id="aegisMalChartRatio"></canvas></div>
				<div class="aegis-mal-sub">
					<span><?php esc_html_e( 'Suspects vs Clean', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['latest']['scan_label'] ); ?></span>
				</div>
			</div>

			<div class="aegis-mal-card">
				<h3>
					<span class="dashicons dashicons-warning"></span>
					<?php esc_html_e( 'Findings by Severity', 'aegisshield-security' ); ?>
					<span class="aegis-mal-pill"><?php echo esc_html( (string) $data['severity']['total'] ); ?></span>
				</h3>
				<div class="aegis-mal-canvas"><canvas id="aegisMalChartSeverity"></canvas></div>
				<div class="aegis-mal-sub">
					<span><?php esc_html_e( 'High / Medium / Low', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['latest']['time_label'] ); ?></span>
				</div>
			</div>

			<div class="aegis-mal-card">
				<h3>
					<span class="dashicons dashicons-book-alt"></span>
					<?php esc_html_e( 'Attack Story Signals', 'aegisshield-security' ); ?>
					<span class="aegis-mal-pill"><?php echo esc_html( (string) $data['signals']['total'] ); ?></span>
				</h3>
				<div class="aegis-mal-canvas"><canvas id="aegisMalChartSignals"></canvas></div>
				<div class="aegis-mal-sub">
					<span><?php esc_html_e( 'Top malware behaviors', 'aegisshield-security' ); ?></span>
					<span><?php echo esc_html( $data['signals']['note'] ); ?></span>
				</div>
			</div>

		</div>

		<script>
		(function(){
			// Load Chart.js once (shared across admin pages).
			if (typeof window.__aegisChartLoaded === 'undefined') {
				window.__aegisChartLoaded = true;
				var s=document.createElement('script');
				s.src=<?php echo wp_json_encode( $chart_src ); ?>;
				s.onload=initMalCharts;
				document.head.appendChild(s);
			} else {
				initMalCharts();
			}

			function initMalCharts() {
				var wrap=document.getElementById('aegis-mal-dash');
				if(!wrap) return;
				var raw=wrap.getAttribute('data-mal');
				if(!raw) return;

				var data;
				try { data=JSON.parse(raw); } catch(e){ return; }
				if(typeof Chart==='undefined') return;

				function ctx(id){ var el=document.getElementById(id); return el?el.getContext('2d'):null; }

				// 1) Scan profile coverage (radar)
				var c1=ctx('aegisMalChartProfile');
				if(c1 && data.profile){
					new Chart(c1,{
						type:'radar',
						data:{
							labels:data.profile.labels,
							datasets:[{ label:'Enabled', data:data.profile.values, fill:true }]
						},
						options:{
							responsive:true, maintainAspectRatio:false,
							plugins:{ legend:{ position:'bottom' } },
							scales:{ r:{ beginAtZero:true, suggestedMax:1, ticks:{ stepSize:1, precision:0 } } }
						}
					});
				}

				// 2) Suspect ratio (doughnut)
				var c2=ctx('aegisMalChartRatio');
				if(c2 && data.latest){
					new Chart(c2,{
						type:'doughnut',
						data:{
							labels:['Suspect','Clean'],
							datasets:[{ data:[data.latest.suspects, data.latest.clean] }]
						},
						options:{ responsive:true, maintainAspectRatio:false, plugins:{ legend:{ position:'bottom' } }, cutout:'65%' }
					});
				}

				// 3) Severity breakdown (bar)
				var c3=ctx('aegisMalChartSeverity');
				if(c3 && data.severity){
					new Chart(c3,{
						type:'bar',
						data:{
							labels:['High','Medium','Low'],
							datasets:[{ data:[data.severity.high, data.severity.medium, data.severity.low] }]
						},
						options:{
							responsive:true, maintainAspectRatio:false,
							plugins:{ legend:{ display:false } },
							scales:{ x:{ grid:{ display:false } }, y:{ beginAtZero:true, ticks:{ precision:0 } } }
						}
					});
				}

				// 4) Attack Story signals (horizontal bar)
				var c4=ctx('aegisMalChartSignals');
				if(c4 && data.signals && data.signals.labels){
					new Chart(c4,{
						type:'bar',
						data:{
							labels:data.signals.labels,
							datasets:[{ label:'Hits', data:data.signals.values }]
						},
						options:{
							indexAxis:'y',
							responsive:true, maintainAspectRatio:false,
							plugins:{ legend:{ display:false } },
							scales:{ x:{ beginAtZero:true, ticks:{ precision:0 } }, y:{ grid:{ display:false } } }
						}
					});
				}
			}
		})();
		</script>
		<?php
	}

	protected function get_malware_dashboard_data() {
		$settings = $this->plugin->get_settings();
		$section  = 'malware';

		$scan_dirs = $settings->get( $section, 'scan_dirs', array( 'plugins', 'themes' ) );
		if ( ! is_array( $scan_dirs ) ) {
			$scan_dirs = array( 'plugins', 'themes' );
		}
		$scan_dirs = array_values( array_unique( array_map( 'sanitize_key', $scan_dirs ) ) );

		// Profile targets (normalize)
		$targets = array(
			'core'      => __( 'Core', 'aegisshield-security' ),
			'plugins'   => __( 'Plugins', 'aegisshield-security' ),
			'themes'    => __( 'Themes', 'aegisshield-security' ),
			'uploads'   => __( 'Uploads', 'aegisshield-security' ),
			'muplugins' => __( 'MU-Plugins', 'aegisshield-security' ),
			'other'     => __( 'Other', 'aegisshield-security' ),
		);

		$profile_labels = array_values( $targets );
		$profile_values = array();
		$enabled_count  = 0;

		foreach ( array_keys( $targets ) as $k ) {
			$on = in_array( $k, $scan_dirs, true ) ? 1 : 0;
			$profile_values[] = $on;
			if ( $on ) { $enabled_count++; }
		}

		// Latest scan meta + results
		$meta = $settings->get( $section, 'last_results_meta', array() );
		if ( ! is_array( $meta ) ) {
			$meta = array();
		}

		$results = $settings->get( $section, 'last_results', array() );
		if ( ! is_array( $results ) ) {
			$results = array();
		}

		$file_count    = (int) ( $meta['file_count'] ?? 0 );
		$suspect_count = (int) ( $meta['suspect_count'] ?? count( $results ) );
		$clean_count   = max( 0, $file_count - $suspect_count );

		$scan_type = (string) ( $meta['scan_type'] ?? '' ); // 'quick' or 'full'
		$partial   = (int) ( $meta['partial'] ?? 0 );
		$completed = (int) ( $meta['completed_at'] ?? 0 );

		// Severity counts from results[*]['level']
		$sev = array( 'high' => 0, 'medium' => 0, 'low' => 0 );
		foreach ( $results as $r ) {
			if ( ! is_array( $r ) ) { continue; }
			$lvl = strtolower( (string) ( $r['level'] ?? '' ) );
			if ( ! isset( $sev[ $lvl ] ) ) { $lvl = 'low'; }
			$sev[ $lvl ]++;
		}

		// Attack Story Signals (categorize reasons into actionable buckets)
		$signals = array(
			__( 'Eval/Base64', 'aegisshield-security' )      => 0,
			__( 'Iframe Inject', 'aegisshield-security' )    => 0,
			__( 'Remote Fetch', 'aegisshield-security' )     => 0,
			__( 'File Write', 'aegisshield-security' )       => 0,
			__( 'Shell/Backdoor', 'aegisshield-security' )   => 0,
			__( 'Obfuscation', 'aegisshield-security' )      => 0,
			__( 'Core Tamper', 'aegisshield-security' )      => 0,
		);

		$bump = function( $reason ) use ( &$signals ) {
			$t = strtolower( (string) $reason );

			if ( strpos( $t, 'core file' ) !== false || strpos( $t, 'checksum' ) !== false ) {
				$signals[ __( 'Core Tamper', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'eval' ) !== false || strpos( $t, 'base64' ) !== false ) {
				$signals[ __( 'Eval/Base64', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'iframe' ) !== false ) {
				$signals[ __( 'Iframe Inject', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'curl' ) !== false || strpos( $t, 'file_get_contents' ) !== false || strpos( $t, 'http' ) !== false || strpos( $t, 'remote' ) !== false ) {
				$signals[ __( 'Remote Fetch', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'fopen' ) !== false || strpos( $t, 'fwrite' ) !== false || strpos( $t, 'file_put_contents' ) !== false || strpos( $t, 'write' ) !== false ) {
				$signals[ __( 'File Write', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'shell' ) !== false || strpos( $t, 'backdoor' ) !== false || strpos( $t, 'system(' ) !== false || strpos( $t, 'exec(' ) !== false || strpos( $t, 'passthru' ) !== false ) {
				$signals[ __( 'Shell/Backdoor', 'aegisshield-security' ) ]++;
				return;
			}
			if ( strpos( $t, 'gzinflate' ) !== false || strpos( $t, 'str_rot13' ) !== false || strpos( $t, 'obfus' ) !== false ) {
				$signals[ __( 'Obfuscation', 'aegisshield-security' ) ]++;
				return;
			}

			// fallback bucket
			$signals[ __( 'Obfuscation', 'aegisshield-security' ) ]++;
		};

		foreach ( $results as $r ) {
			if ( ! is_array( $r ) ) { continue; }
			$reasons = isset( $r['reasons'] ) && is_array( $r['reasons'] ) ? $r['reasons'] : array();
			foreach ( $reasons as $reason ) {
				$bump( $reason );
			}
		}

		arsort( $signals );
		$top = array_slice( $signals, 0, 6, true );

		$time_label = $completed ? date_i18n( 'M j, Y g:ia', $completed ) : __( 'No scan yet', 'aegisshield-security' );

		$scan_label = __( 'Unknown', 'aegisshield-security' );
		if ( $scan_type === 'quick' ) {
			$scan_label = $partial ? __( 'Quick (Partial)', 'aegisshield-security' ) : __( 'Quick', 'aegisshield-security' );
		} elseif ( $scan_type === 'full' ) {
			$scan_label = $partial ? __( 'Full (Partial)', 'aegisshield-security' ) : __( 'Full', 'aegisshield-security' );
		}

		return array(
			'profile' => array(
				'labels'        => $profile_labels,
				'values'        => $profile_values,
				'enabled_count' => $enabled_count,
				/* translators: %d: number of targets enabled in the scan profile. */
				'mode_label'    => sprintf( __( '%d targets', 'aegisshield-security' ), $enabled_count ),
			),
			'latest' => array(
				'files'      => $file_count,
				'suspects'   => $suspect_count,
				'clean'      => $clean_count,
				'scan_label' => $scan_label,
				'time_label' => $time_label,
			),
			'severity' => array(
				'high'   => (int) $sev['high'],
				'medium' => (int) $sev['medium'],
				'low'    => (int) $sev['low'],
				'total'  => (int) ( $sev['high'] + $sev['medium'] + $sev['low'] ),
			),
			'signals' => array(
				'labels' => array_keys( $top ),
				'values' => array_values( $top ),
				'total'  => (int) array_sum( $signals ),
				'note'   => $suspect_count ? __( 'From latest findings', 'aegisshield-security' ) : __( 'No findings yet', 'aegisshield-security' ),
			),
		);
	}
}
