<?php
namespace AegisShield\Admin_Pages\Malware;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Page_Malware_Tab_Incremental {

    public static function render_tab( AS_Plugin $plugin ) {
        $settings = $plugin->get_settings();
        $section  = 'malware';

        $is_pro  = function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
        $message = '';

        if ( $is_pro && isset( $_POST['as_ms_incremental_nonce'] ) && is_string( $_POST['as_ms_incremental_nonce'] ) ) {
            check_admin_referer( 'as_ms_incremental_settings', 'as_ms_incremental_nonce' );

            if ( current_user_can( 'manage_options' ) ) {
                $quick_enabled = isset( $_POST['schedule_quick_enabled'] ) ? 'on' : 'off';
                $settings->set( $section, 'schedule_quick_enabled', $quick_enabled );

                // Frequency.
                $allowed_frequencies = array( 'daily', 'weekly', 'monthly', 'quarterly' );
                $freq                = isset( $_POST['quick_schedule_frequency'] )
                    ? sanitize_text_field( wp_unslash( $_POST['quick_schedule_frequency'] ) )
                    : 'daily';

                if ( ! in_array( $freq, $allowed_frequencies, true ) ) {
                    $freq = 'daily';
                }

                $settings->set( $section, 'quick_schedule_frequency', $freq );
                $settings->save();

                // Manual "Run Quick Scan Now".
                if ( isset( $_POST['as_ms_run_quick_scan_now'] ) ) {
                    $module = new \AegisShield\Modules\AS_Module_Malware( $plugin );
                    $module->run_incremental_scan( 'manual_quick' );
                    $message = __( 'Incremental quick scan settings saved and a new quick scan has been run.', 'aegisshield-security' );
                } else {
                    $message = __( 'Incremental quick scan schedule settings saved.', 'aegisshield-security' );
                }
            }
        }

        // Reload schedule values for display.
        $quick_enabled_val  = (string) $settings->get( $section, 'schedule_quick_enabled', '' );
        $quick_enabled_bool = in_array( $quick_enabled_val, array( 'on', '1', 'yes' ), true );

        $freq = (string) $settings->get( $section, 'quick_schedule_frequency', 'daily' );
        if ( ! in_array( $freq, array( 'daily', 'weekly', 'monthly', 'quarterly' ), true ) ) {
            $freq = 'daily';
        }

        // Fetch metadata and results as persisted by the malware module.
        $meta    = $settings->get( $section, 'last_results_meta', array() );
        $results = $settings->get( $section, 'last_results', array() );

        ?>
        <div class="as-ms-tab as-ms-tab-incremental">
            <h2><?php esc_html_e( 'Incremental Quick Scan (Pro)', 'aegisshield-security' ); ?></h2>

            <?php if ( $message ) : ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo esc_html( $message ); ?></p>
                </div>
            <?php endif; ?>

            <?php if ( ! $is_pro ) : ?>
                <p class="description">
                    <?php esc_html_e(
                        'Incremental “quick scan” mode is available in AegisShield Pro. Activate your license under AegisShield → License & Upgrades to unlock this feature.',
                        'aegisshield-security'
                    ); ?>
                </p>
                <?php return; ?>
            <?php endif; ?>

            <p class="description">
                <?php esc_html_e(
                    'Configure how often incremental quick scans should run based on recently changed files, and launch an on-demand quick scan when you need an immediate check.',
                    'aegisshield-security'
                ); ?>
            </p>

            <h3><?php esc_html_e( 'Incremental Quick Scan Schedule', 'aegisshield-security' ); ?></h3>

            <form method="post">
                <?php wp_nonce_field( 'as_ms_incremental_settings', 'as_ms_incremental_nonce' ); ?>

                <p>
                    <label>
                        <input type="checkbox" name="schedule_quick_enabled" value="1" <?php checked( $quick_enabled_bool ); ?> />
                        <?php esc_html_e( 'Enable automatic incremental quick scans', 'aegisshield-security' ); ?>
                    </label>
                </p>

                <p>
                    <strong><?php esc_html_e( 'How often should incremental quick scans run?', 'aegisshield-security' ); ?></strong><br />
                    <label>
                        <input type="radio" name="quick_schedule_frequency" value="daily" <?php checked( $freq, 'daily' ); ?> />
                        <?php esc_html_e( 'Once a day', 'aegisshield-security' ); ?>
                    </label><br />
                    <label>
                        <input type="radio" name="quick_schedule_frequency" value="weekly" <?php checked( $freq, 'weekly' ); ?> />
                        <?php esc_html_e( 'Once a week', 'aegisshield-security' ); ?>
                    </label><br />
                    <label>
                        <input type="radio" name="quick_schedule_frequency" value="monthly" <?php checked( $freq, 'monthly' ); ?> />
                        <?php esc_html_e( 'Once a month', 'aegisshield-security' ); ?>
                    </label><br />
                    <label>
                        <input type="radio" name="quick_schedule_frequency" value="quarterly" <?php checked( $freq, 'quarterly' ); ?> />
                        <?php esc_html_e( 'Once a quarter', 'aegisshield-security' ); ?>
                    </label>
                </p>

                <p>
                    <button type="submit" name="as_ms_incremental_save" class="button button-primary">
                        <?php esc_html_e( 'Save schedule', 'aegisshield-security' ); ?>
                    </button>
                    <button type="submit" name="as_ms_run_quick_scan_now" class="button">
                        <?php esc_html_e( 'Run Quick Scan Now', 'aegisshield-security' ); ?>
                    </button>
                </p>
            </form>

            <hr />

            <?php
            $has_meta = is_array( $meta ) && ! empty( $meta );

            if ( ! $has_meta ) :
                ?>
                <p class="description">
                    <?php esc_html_e( 'No incremental quick scans have been run yet.', 'aegisshield-security' ); ?>
                </p>
                <?php
                return;
            endif;

            $file_count    = isset( $meta['file_count'] ) ? (int) $meta['file_count'] : 0;
            $suspect_count = isset( $meta['suspect_count'] ) ? (int) $meta['suspect_count'] : 0;

            if ( 0 === $file_count ) :
                ?>
                <p class="description">
                    <?php esc_html_e( 'No changed files were detected during the last incremental quick scan.', 'aegisshield-security' ); ?>
                </p>
                <?php
                return;
            endif;
            ?>

            <h3><?php esc_html_e( 'Last Incremental Quick Scan Summary', 'aegisshield-security' ); ?></h3>

            <table class="widefat fixed striped">
                <tbody>
                    <tr>
                        <th><?php esc_html_e( 'Scan Type:', 'aegisshield-security' ); ?></th>
                        <td><?php echo isset( $meta['scan_type'] ) ? esc_html( $meta['scan_type'] ) : ''; ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Started At:', 'aegisshield-security' ); ?></th>
                        <td>
                            <?php
                            $started = isset( $meta['started_at'] ) ? (int) $meta['started_at'] : 0;
                            echo $started ? esc_html( gmdate( 'Y-m-d H:i:s', $started ) ) : '';
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Completed At:', 'aegisshield-security' ); ?></th>
                        <td>
                            <?php
                            $completed = isset( $meta['completed_at'] ) ? (int) $meta['completed_at'] : 0;
                            echo $completed ? esc_html( gmdate( 'Y-m-d H:i:s', $completed ) ) : '';
                            ?>
                        </td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Files Scanned:', 'aegisshield-security' ); ?></th>
                        <td><?php echo esc_html( $file_count ); ?></td>
                    </tr>
                    <tr>
                        <th><?php esc_html_e( 'Suspect Files:', 'aegisshield-security' ); ?></th>
                        <td><?php echo esc_html( $suspect_count ); ?></td>
                    </tr>
                </tbody>
            </table>

            <h3 style="margin-top:25px;"><?php esc_html_e( 'Scan Results', 'aegisshield-security' ); ?></h3>

            <table class="widefat fixed striped as-sortable-table" id="as-incremental-results-table">
                <thead>
                    <tr>
                        <th data-sort="file"><?php esc_html_e( 'File', 'aegisshield-security' ); ?></th>
                        <th data-sort="score"><?php esc_html_e( 'Score', 'aegisshield-security' ); ?></th>
                        <th data-sort="level"><?php esc_html_e( 'Level', 'aegisshield-security' ); ?></th>
                        <th data-sort="scan_time"><?php esc_html_e( 'Scan Time', 'aegisshield-security' ); ?></th>
                        <th><?php esc_html_e( 'Reasons', 'aegisshield-security' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( is_array( $results ) && ! empty( $results ) ) : ?>
                        <?php foreach ( $results as $row ) : ?>
                            <tr>
                                <td><?php echo isset( $row['file'] ) ? esc_html( $row['file'] ) : ''; ?></td>
                                <td><?php echo isset( $row['score'] ) ? esc_html( (int) $row['score'] ) : 0; ?></td>
                                <td><?php echo isset( $row['level'] ) ? esc_html( ucfirst( $row['level'] ) ) : ''; ?></td>
                                <td>
                                    <?php
                                    $scan_time = isset( $row['scan_time'] ) ? (int) $row['scan_time'] : 0;
                                    echo $scan_time ? esc_html( gmdate( 'Y-m-d H:i:s', $scan_time ) ) : '';
                                    ?>
                                </td>
                                <td>
                                    <?php
                                    $reasons = isset( $row['reasons'] ) && is_array( $row['reasons'] ) ? $row['reasons'] : array();
                                    ?>
                                    <button class="as-reasons-toggle" type="button" aria-expanded="false">▼</button>
                                    <div class="as-reasons-content" style="display:none;margin-top:5px;">
                                        <?php if ( ! empty( $reasons ) ) : ?>
                                            <ul>
                                                <?php foreach ( $reasons as $reason ) : ?>
                                                    <li><?php echo esc_html( $reason ); ?></li>
                                                <?php endforeach; ?>
                                            </ul>
                                        <?php else : ?>
                                            <em><?php esc_html_e( 'No specific reasons recorded.', 'aegisshield-security' ); ?></em>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr>
                            <td colspan="5">
                                <?php esc_html_e( 'No suspicious files were flagged during the last incremental quick scan.', 'aegisshield-security' ); ?>
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <script>
                (function() {
                    // Column Sorting
                    const table   = document.getElementById('as-incremental-results-table');
                    if (!table) {
                        return;
                    }
                    const headers = table.querySelectorAll('th[data-sort]');
                    let sortDirection = 1;

                    headers.forEach(header => {
                        header.addEventListener('click', () => {
                            const columnIndex = header.cellIndex;
                            const tbody       = table.querySelector('tbody');
                            const rows        = Array.from(tbody.querySelectorAll('tr'));

                            rows.sort((a, b) => {
                                const aCell = a.cells[columnIndex].innerText.toLowerCase();
                                const bCell = b.cells[columnIndex].innerText.toLowerCase();

                                const aNum = parseFloat(aCell);
                                const bNum = parseFloat(bCell);

                                if (!isNaN(aNum) && !isNaN(bNum)) {
                                    return (aNum - bNum) * sortDirection;
                                }

                                return aCell.localeCompare(bCell) * sortDirection;
                            });

                            sortDirection *= -1;

                            tbody.innerHTML = '';
                            rows.forEach(row => tbody.appendChild(row));
                        });
                    });

                    // Reasons Toggle
                    document.querySelectorAll('.as-reasons-toggle').forEach(btn => {
                        btn.addEventListener('click', () => {
                            const content  = btn.nextElementSibling;
                            const expanded = btn.getAttribute('aria-expanded') === 'true';

                            btn.setAttribute('aria-expanded', expanded ? 'false' : 'true');
                            btn.textContent = expanded ? '▼' : '▲';
                            content.style.display = expanded ? 'none' : 'block';
                        });
                    });
                })();
            </script>
        </div>
        <?php
    }
}