<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Module_DB_Tools implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'db_tools';
    }

    public function register_settings() {
    }

    public function init() {

        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_run_weekly_optimization' ) );
        add_action( 'aegisshield_hourly_maintenance', array( $this, 'maybe_check_growth' ) );
    }

/**
 * Get WP_Filesystem instance (initializes if needed).
 *
 * @return \WP_Filesystem_Base|null
 */
protected function get_filesystem() {
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }

    global $wp_filesystem;
    if ( ! $wp_filesystem ) {
        WP_Filesystem();
    }

    return $wp_filesystem;
}

/**
 * Sanitize a MySQL identifier (table/column name). Identifiers cannot be parameterized in $wpdb->prepare(),
 * so we strictly whitelist characters and backtick-escape.
 *
 * @param string $ident Identifier.
 * @return string Sanitized identifier or empty string if invalid.
 */
protected function sanitize_db_identifier( $ident ) {
    $ident = (string) $ident;
    $ident = trim( str_replace( '`', '', $ident ) );

    if ( '' === $ident ) {
        return '';
    }

    // Only allow letters, numbers, and underscores.
    if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $ident ) ) {
        return '';
    }

    return $ident;
}


/**
 * Sanitize a database prefix (identifier-safe).
 *
 * @param string $prefix Prefix.
 * @return string Sanitized prefix or empty string if invalid.
 */
protected function sanitize_db_prefix( $prefix ) {
    $prefix = (string) $prefix;
    $prefix = trim( str_replace( '`', '', $prefix ) );

    if ( '' === $prefix ) {
        return '';
    }

    // Only allow letters, numbers, and underscores.
    if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $prefix ) ) {
        return '';
    }

    return $prefix;
}

/**
 * Escape an SQL identifier (strict allowlist + backticks).
 *
 * @param string $ident Identifier.
 * @return string Backticked identifier or empty string if invalid.
 */
protected function escape_sql_identifier( $ident ) {
    $ident = $this->sanitize_db_identifier( $ident );
    if ( '' === $ident ) {
        return '';
    }

    return '`' . $ident . '`';
}

/**
 * Replace {{table}} token in SQL with a safely escaped identifier.
 *
 * @param string $sql SQL containing {{table}} token.
 * @param string $table Table name.
 * @return string SQL with token replaced (or empty string on invalid).
 */
protected function sql_with_table( $sql, $table ) {
    $escaped = $this->escape_sql_identifier( $table );
    if ( '' === $escaped ) {
        return '';
    }

    return str_replace( '{{table}}', $escaped, (string) $sql );
}


/**
 * Wrap a sanitized identifier in backticks.
 *
 * @param string $ident Identifier.
 * @return string Backticked identifier or empty string.
 */
protected function backtick_ident( $ident ) {
    $ident = $this->sanitize_db_identifier( $ident );
    if ( '' === $ident ) {
        return '';
    }

    return '`' . $ident . '`';
}


	protected function acquire_run_lock( $key, $ttl = 3300 ) {
		$key = 'as_db_tools_lock_' . sanitize_key( (string) $key );

		// If lock exists, skip.
		if ( get_transient( $key ) ) {
			return false;
		}

		// Set lock (default ~55 min).
		set_transient( $key, 1, (int) $ttl );
		return true;
	}

	public function log_violation( $event, $message_or_context = '', $context = array(), $level = 'info' ) {
		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( ! $logger ) {
			return;
		}

		// Allow signature: log_violation( 'event', array( ...context... ) )
		if ( is_array( $message_or_context ) && empty( $context ) ) {
			$context            = $message_or_context;
			$message_or_context = '';
		}

		if ( ! is_array( $context ) ) {
			$context = array();
		}

		// Always tag this module and severity.
		$context = array_merge(
			array(
				'module' => 'db_tools',
				'level'  => (string) $level,
			),
			$context
		);

		// Attach user_id when available.
		if ( function_exists( 'get_current_user_id' ) && empty( $context['user_id'] ) ) {
			$context['user_id'] = (int) get_current_user_id();
		}

		// Use whatever logger signature your system supports (your code already uses both forms).
		if ( is_string( $message_or_context ) && '' !== $message_or_context ) {
			// (event, message, context)
			$logger->log( $event, $message_or_context, $context );
			return;
		}

		// (event, context)
		$logger->log( $event, $context );
	}

    public function maybe_run_weekly_optimization() {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';
		
		$threshold_mb = (int) $settings->get( $section, 'growth_threshold_mb', 100 );
		
		$growth_email           = trim( (string) $settings->get( $section, 'growth_email', '' ) );
		$growth_email_addresses = trim( (string) $settings->get( $section, 'growth_email_addresses', '' ) );

		$recipients = array();

		if ( '' !== $growth_email ) {
			$recipients[] = $growth_email;
		}

		if ( '' !== $growth_email_addresses ) {
			$parts = preg_split( '/[\s,;]+/', $growth_email_addresses );
			if ( is_array( $parts ) ) {
				foreach ( $parts as $p ) {
					$p = sanitize_email( $p );
					if ( $p && is_email( $p ) ) {
						$recipients[] = $p;
					}
				}
			}
		}

		$recipients = array_values( array_unique( array_filter( $recipients ) ) );

		if ( empty( $recipients ) ) {
			$notifier = $this->plugin->get_notifier();
			if ( $notifier && method_exists( $notifier, 'get_global_recipients' ) ) {
				$recipients = (array) $notifier->get_global_recipients();
				$recipients = array_values( array_unique( array_filter( $recipients ) ) );
			}
		}

		if ( empty( $recipients ) ) {
			$recipients = array( get_option( 'admin_email' ) );
		}

		$weekly = (string) $settings->get( $section, 'weekly_optimize', '' );
		if ( 'on' !== $weekly ) {
			// Do not log heartbeat/disabled checks (creates massive noise).
			return;
		}

        $last = (int) $settings->get( $section, 'last_optimize', 0 );
        $now  = time();

        if ( $last && ( $now - $last ) < WEEK_IN_SECONDS ) {
            return;
        }

        $this->run_optimization( 'auto' );

        $settings->set( $section, 'last_optimize', $now );
        $settings->save();
    }

	public function run_manual_optimization() {
		if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
			$this->log_violation(
				'db_tools_manual_optimize_blocked',
				__( 'Manual optimization blocked: insufficient permissions.', 'aegisshield-security' ),
				array( 'context' => 'manual', 'reason' => 'permissions' ),
				'critical'
			);
			return;
		}

		$this->log_violation(
			'db_tools_manual_optimize_requested',
			__( 'Manual optimization requested.', 'aegisshield-security' ),
			array( 'context' => 'manual' ),
			'info'
		);

		$this->run_optimization( 'manual' );
	}

    protected function run_optimization( $context ) {
        global $wpdb;

        $logger = $this->plugin->get_logger();
        $prefix = $wpdb->prefix;
        $like   = $wpdb->esc_like( $prefix ) . '%';

        $cache_key = 'as_dbtools_tables_' . md5( $like );
        $tables    = wp_cache_get( $cache_key, 'aegisshield' );
        if ( false === $tables ) {
            // This query has no core wrapper; cache to avoid repeated expensive scans during maintenance.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( is_array( $tables ) ) {
                wp_cache_set( $cache_key, $tables, 'aegisshield', HOUR_IN_SECONDS );
            }
        }

        if ( empty( $tables ) ) {
            return;
        }

        foreach ( $tables as $table ) {
            $table = $this->sanitize_db_identifier( $table );
            if ( '' === $table ) {
                continue;
            }
            // OPTIMIZE TABLE is safe for InnoDB/MyISAM and is non-destructive.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
            $result = $wpdb->query( $wpdb->prepare( 'OPTIMIZE TABLE %i', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.

			if ( false === $result ) {
				$this->log_violation(
					'db_tools_optimize_table_failed',
					__( 'OPTIMIZE TABLE failed.', 'aegisshield-security' ),
					array(
						'context'     => $context,
						'table'       => $table,
						'wpdb_error'  => isset( $wpdb->last_error ) ? (string) $wpdb->last_error : '',
					),
					'error'
				);
			} else {
				$this->log_violation(
					'db_tools_optimize_table_ok',
					__( 'OPTIMIZE TABLE succeeded.', 'aegisshield-security' ),
					array(
						'context' => $context,
						'table'   => $table,
					),
					'debug'
				);
			}
        }

		$this->log_violation(
			'db_tools_optimized',
			sprintf(
				/* translators: %s: context (manual/auto). */
				__( 'Database tables optimized (%s).', 'aegisshield-security' ),
				$context
			),
			array(
				'context' => $context,
				'tables'  => $tables,
			),
			'info'
		);
    }

    public function maybe_check_growth() {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

		$monitoring = (string) $settings->get( $section, 'growth_monitoring', '' );
		if ( 'on' !== $monitoring ) {
			// Do not log heartbeat/disabled checks (creates massive noise).
			return;
		}

        $summary = $this->get_table_summary();
        $warnings = isset( $summary['warnings'] ) ? $summary['warnings'] : array();
		if ( empty( $warnings ) ) {
			// No alert-worthy growth: do not log heartbeat.
			return;
		}

		$growth_email           = trim( (string) $settings->get( $section, 'growth_email', '' ) );
		$growth_email_addresses = trim( (string) $settings->get( $section, 'growth_email_addresses', '' ) );

		if ( '' === $growth_email && '' === $growth_email_addresses ) {
			$notifier = $this->plugin->get_notifier();
			$global   = ( $notifier && method_exists( $notifier, 'get_global_recipients' ) )
				? (array) $notifier->get_global_recipients()
				: array();

			if ( empty( $global ) ) {
				return;
			}
		}

        $last_email = (int) $settings->get( $section, 'last_growth_email', 0 );
        $now        = time();

        // Avoid spamming; at most one email every 12 hours.
        if ( $last_email && ( $now - $last_email ) < 12 * HOUR_IN_SECONDS ) {
            return;
        }

        $tables = isset( $summary['tables'] ) ? $summary['tables'] : array();
		$this->log_violation(
			'db_table_growth_detected',
			__( 'Database table growth detected.', 'aegisshield-security' ),
			array(
				'warning_count' => is_array( $warnings ) ? count( $warnings ) : 0,
			),
			'warning'
		);
        $this->send_growth_email( $warnings, $tables );
		$this->log_violation(
			'db_tools_growth_email_sent',
			__( 'Growth alert email sent.', 'aegisshield-security' ),
			array(),
			'info'
		);

        $settings->set( $section, 'last_growth_email', $now );
        $settings->save();
    }

    protected function send_growth_email( array $warnings, array $tables ) {
        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

        $to_raw = (string) $settings->get( $section, 'growth_email_addresses', '' );
        $recipients = array();

        if ( '' !== $to_raw ) {
            $parts = array_map( 'trim', explode( ',', $to_raw ) );
            foreach ( $parts as $addr ) {
                if ( is_email( $addr ) ) {
                    $recipients[] = $addr;
                }
            }
        }

        if ( empty( $recipients ) ) {
            $admin_email = get_option( 'admin_email' );
            if ( $admin_email && is_email( $admin_email ) ) {
                $recipients[] = $admin_email;
            }
        }

        if ( empty( $recipients ) ) {
            return;
        }

        $site_name = wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES );
        $subject   = sprintf(
            /* translators: %s: site name. */
            __( '[AegisShield] Database growth warning on %s', 'aegisshield-security' ),
            $site_name
        );

        // Build a brief body with warnings and top heavy tables.
        $body_lines   = array();
        $body_lines[] = __( 'AegisShield DB Tools has detected significant database table growth or other warnings:', 'aegisshield-security' );
        $body_lines[] = '';
        foreach ( $warnings as $warning ) {
            $body_lines[] = ' - ' . wp_strip_all_tags( $warning );
        }

        $body_lines[] = '';
        $body_lines[] = __( 'Largest tables (by total size):', 'aegisshield-security' );

        // Sort tables by total size descending.
        usort(
            $tables,
            function ( $a, $b ) {
                $at = isset( $a['total_bytes'] ) ? (int) $a['total_bytes'] : 0;
                $bt = isset( $b['total_bytes'] ) ? (int) $b['total_bytes'] : 0;
                if ( $at === $bt ) {
                    return 0;
                }
                return ( $at > $bt ) ? -1 : 1;
            }
        );

        $max_tables = 5;
        $count      = 0;
        foreach ( $tables as $table ) {
            $name  = isset( $table['name'] ) ? $table['name'] : '';
            $total = isset( $table['total_bytes'] ) ? (int) $table['total_bytes'] : 0;
            $size  = size_format( $total );
            $body_lines[] = sprintf( ' - %s (%s)', $name, $size );
            $count++;
            if ( $count >= $max_tables ) {
                break;
            }
        }

        $body_lines[] = '';
        $body_lines[] = __( 'You can review detailed DB statistics in AegisShield → DB Tools.', 'aegisshield-security' );

		$body = implode( "\n", $body_lines );

		$headers  = array( 'Content-Type: text/plain; charset=UTF-8' );
		$notifier = $this->plugin->get_notifier();

		if ( $notifier && method_exists( $notifier, 'send_email_to' ) ) {
			$notifier->send_email_to( $recipients, $subject, $body, $headers );
		} else {
			wp_mail( implode( ',', $recipients ), $subject, $body, $headers );
		}
		
        $this->plugin->get_logger()->log(
            'db_table_growth_detected',
            __( 'Database table growth detected.', 'aegisshield-security' ),
            'medium',
			array(
				'top_tables'    => $body_lines,
				'threshold_mb'  => $threshold_mb,
				'recipients'    => count( $recipients ),
			)
        );
    }

    public function get_table_summary() {
        global $wpdb;

        $settings = $this->plugin->get_settings();
        $section  = 'db_tools';

        $prefix = $wpdb->prefix;
        $like   = $wpdb->esc_like( $prefix ) . '%';

        $cache_key  = 'as_dbtools_table_status_' . md5( $like );
        $status_rows = wp_cache_get( $cache_key, 'aegisshield' );
        if ( false === $status_rows ) {
            // SHOW TABLE STATUS is relatively heavy; cache briefly for admin views and hourly checks.
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $status_rows = $wpdb->get_results( $wpdb->prepare( 'SHOW TABLE STATUS LIKE %s', $like ), ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( is_array( $status_rows ) ) {
                wp_cache_set( $cache_key, $status_rows, 'aegisshield', 5 * MINUTE_IN_SECONDS );
            }
        }


        $previous_snapshot = $settings->get( $section, 'last_table_snapshot', array() );
        if ( ! is_array( $previous_snapshot ) ) {
            $previous_snapshot = array();
        }

        $snapshot = array();
        $tables   = array();
        $spikes   = array();

        if ( $status_rows ) {
            foreach ( $status_rows as $row ) {
                $name   = isset( $row['Name'] ) ? $row['Name'] : '';
                $engine = isset( $row['Engine'] ) ? $row['Engine'] : '';
                $rows   = isset( $row['Rows'] ) ? (int) $row['Rows'] : 0;
                $data   = isset( $row['Data_length'] ) ? (int) $row['Data_length'] : 0;
                $index  = isset( $row['Index_length'] ) ? (int) $row['Index_length'] : 0;

                $total = $data + $index;

                $prev_size = 0;
                if ( isset( $previous_snapshot[ $name ]['size'] ) ) {
                    $prev_size = (int) $previous_snapshot[ $name ]['size'];
                }

                $delta     = $total - $prev_size;
                $delta_pct = ( $prev_size > 0 && $total > 0 ) ? ( ( $total / $prev_size ) - 1 ) : 0.0;

                if ( $prev_size > 0 && $delta > 5 * 1024 * 1024 && $delta_pct >= 1.0 ) {
                    // Grew by more than ~100% and at least 5MB.
                    $spikes[] = $name;
                }

                $tables[] = array(
                    'name'        => $name,
                    'engine'      => $engine,
                    'rows'        => $rows,
                    'data_bytes'  => $data,
                    'index_bytes' => $index,
                    'total_bytes' => $total,
                    'delta_bytes' => $delta,
                    'delta_pct'   => $delta_pct,
                );

                $snapshot[ $name ] = array(
                    'size' => $total,
                    'rows' => $rows,
                );
            }
        }

        // Save new snapshot for future comparisons.
        $settings->set( $section, 'last_table_snapshot', $snapshot );
        $settings->save();

        $warnings = array();

        // Warn if prefix is still default wp_.
        if ( 'wp_' === $prefix ) {
            $warnings[] = __( 'Your database table prefix is still the default "wp_". Consider using a custom prefix for a small hardening benefit.', 'aegisshield-security' );
        }

        // Warn on significant growth spikes.
        if ( ! empty( $spikes ) ) {
            $warnings[] = sprintf(
                /* translators: %s: comma-separated table names. */
                __( 'The following tables have grown significantly since the last snapshot: %s. This may indicate log or cache bloat.', 'aegisshield-security' ),
                implode( ', ', $spikes )
            );
        }

        return array(
            'tables'   => $tables,
            'warnings' => $warnings,
        );
    }


public function create_db_backup_snapshot() {
    global $wpdb;

    $fs = $this->get_filesystem();
    if ( ! $fs ) {
        return array(
            'success' => false,
            'message' => __( 'Filesystem is not available.', 'aegisshield-security' ),
        );
    }

    $dir = $this->get_db_tools_backup_dir();
    if ( empty( $dir ) || ! is_dir( $dir ) || ! $fs->is_writable( $dir ) ) {
        return array(
            'success' => false,
            'message' => __( 'Backup directory is not writable.', 'aegisshield-security' ),
        );
    }

    $timestamp = gmdate( 'Y-m-d_H-i-s' );
    $db_name   = defined( 'DB_NAME' ) ? (string) DB_NAME : 'database';
    $prefix    = isset( $wpdb->prefix ) ? (string) $wpdb->prefix : '';

    $safe_db   = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $db_name );
    $safe_pref = preg_replace( '/[^A-Za-z0-9_\-]/', '_', $prefix );

    $filename = 'aegisshield-db-backup-' . $safe_db . '-' . $safe_pref . '-' . $timestamp . '.sql';
    $path     = trailingslashit( $dir ) . $filename;

    // Export tables.
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
    $tables = $wpdb->get_col( 'SHOW TABLES' ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
    if ( empty( $tables ) ) {
        return array(
            'success' => false,
            'message' => __( 'No tables found to back up.', 'aegisshield-security' ),
        );
    }

    $out  = "-- AegisShield DB Tools Backup\n";
    $out .= '-- Created (UTC): ' . gmdate( 'c' ) . "\n";
    $out .= '-- Database: ' . $safe_db . "\n";
    $out .= '-- Prefix: ' . $safe_pref . "\n\n";

    foreach ( $tables as $table_raw ) {
        $table = $this->sanitize_db_identifier( $table_raw );
        if ( '' === $table ) {
            continue;
        }

        $table_bt = $this->backtick_ident( $table );
        if ( '' === $table_bt ) {
            continue;
        }

        // Structure.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.NotPrepared
        $create_row = $wpdb->get_row( $wpdb->prepare( 'SHOW CREATE TABLE %i', $table ), ARRAY_N ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( empty( $create_row ) || empty( $create_row[1] ) ) {
            continue;
        }

        $out .= "\n-- ----------------------------------------\n";
        $out .= '-- Table: ' . $table . "\n";
        $out .= 'DROP TABLE IF EXISTS ' . $table_bt . ";\n";
        $out .= $create_row[1] . ";\n\n";

        // Data in chunks to reduce memory.
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
        $row_count = (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM %i', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( $row_count <= 0 ) {
            continue;
        }

        $limit  = 500;
        $offset = 0;

        while ( $offset < $row_count ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Table identifier is sanitized earlier into $table_bt; LIMIT uses placeholders via $wpdb->prepare().
            $results = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                $wpdb->prepare(
                    'SELECT * FROM %i LIMIT %d, %d',
                    $table,
                    (int) $offset,
                    (int) $limit
                ),
                ARRAY_A
            );


            if ( empty( $results ) ) {
                break;
            }

            foreach ( $results as $row ) {
                $columns = array();
                $values  = array();

                foreach ( $row as $col => $val ) {
                    $col_s = $this->sanitize_db_identifier( (string) $col );
                    if ( '' === $col_s ) {
                        continue;
                    }
                    $columns[] = '`' . $col_s . '`';
                    $values[]  = $this->sql_escape_value( $val );
                }

                if ( empty( $columns ) ) {
                    continue;
                }

                $out .= 'INSERT INTO ' . $table_bt . ' (' . implode( ',', $columns ) . ') VALUES (' . implode( ',', $values ) . ");\n";
            }

            $offset += $limit;
        }
    }

    $written = $fs->put_contents( $path, $out, FS_CHMOD_FILE );
    if ( ! $written ) {
        return array(
            'success' => false,
            'message' => __( 'Unable to write backup file.', 'aegisshield-security' ),
        );
    }

    $size   = $fs->size( $path );
    $size_h = $size ? size_format( (int) $size, 2 ) : '';

    $this->log_violation(
        'db_backup_created',
        __( 'Database backup snapshot created.', 'aegisshield-security' ),
        array( /* keep your existing context array here */ ),
        'info'
    );

    return array(
        'success' => true,
        'message' => sprintf(
            /* translators: 1: file name, 2: size */
            __( 'Backup created: %1$s %2$s', 'aegisshield-security' ),
            $filename,
            $size_h ? '(' . $size_h . ')' : ''
        ),
        'file'    => $path,
    );
}

    public function list_db_backups() {
        $dir = $this->get_db_tools_backup_dir();
        if ( empty( $dir ) || ! is_dir( $dir ) ) {
            return array();
        }

        $files = glob( trailingslashit( $dir ) . '*.sql' );
        if ( empty( $files ) ) {
            return array();
        }

        rsort( $files );

        $out = array();
        foreach ( $files as $path ) {
            $path = (string) $path;
            if ( ! file_exists( $path ) ) {
                continue;
            }

            $out[] = array(
                'date' => gmdate( 'Y-m-d H:i:s', (int) filemtime( $path ) ) . ' UTC',
                'file' => basename( $path ),
                'size' => size_format( (int) filesize( $path ), 2 ),
                'path' => $path,
            );
        }

        return $out;
    }

    protected function get_db_tools_backup_dir() {
        $upload = wp_upload_dir();
        $base   = isset( $upload['basedir'] ) ? (string) $upload['basedir'] : '';
        if ( empty( $base ) ) {
            return '';
        }

        $dir = trailingslashit( $base ) . 'aegisshield-backups/db-tools';

        if ( ! file_exists( $dir ) ) {
            wp_mkdir_p( $dir );
        }

        // Best-effort hardening: prevent browsing.
        $index = trailingslashit( $dir ) . 'index.html';
        if ( ! file_exists( $index ) ) {
            $fs = $this->get_filesystem();
            if ( $fs ) {
                $fs->put_contents( $index, '', FS_CHMOD_FILE );
            }
        }

        $htaccess = trailingslashit( $dir ) . '.htaccess';
        if ( ! file_exists( $htaccess ) ) {
            $fs = $this->get_filesystem();
            if ( $fs ) {
                $fs->put_contents( $htaccess, "Deny from all\n", FS_CHMOD_FILE );
            }

        }
        return $dir;
    }

protected function sql_escape_value( $value ) {
    global $wpdb;

    if ( is_null( $value ) ) {
        return 'NULL';
    }

    if ( is_bool( $value ) ) {
        return $value ? '1' : '0';
    }

    if ( is_int( $value ) || is_float( $value ) ) {
        return (string) $value;
    }

    // Use $wpdb->prepare() to safely quote and escape strings.
    return $wpdb->prepare( '%s', (string) $value );
}


    public function apply_prefix_change( $tables, $prefix_current, $prefix_new, $prefix_mode, $exec_mode, $backup_file = '' ) {
        global $wpdb;

        $mode = (string) $prefix_mode;
		if ( $mode !== 'all' ) {
			$this->log_violation(
				'db_prefix_change_blocked',
				__( 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.', 'aegisshield-security' ),
				array(
					'prefix_mode' => (string) $prefix_mode,
					'exec_mode'   => (string) $exec_mode,
				),
				'critical'
			);

			return array(
				'status'  => 'error',
				'message' => 'Unsafe prefix mode blocked. Only full-prefix migrations are allowed.',
			);
		}
		
        if ( function_exists( 'current_user_can' ) && ! current_user_can( 'manage_options' ) ) {
            $this->plugin->get_logger()->log(
                'db_prefix_change_failed',
                __( 'DB prefix change failed: insufficient permissions.', 'aegisshield-security' ),
                'critical',
                array( 'old_prefix' => (string) $prefix_current, 'new_prefix' => (string) $prefix_new, 'reason' => 'permissions', 'user_id' => get_current_user_id() )
            );
            return array(
                'success' => false,
                'message' => __( 'Insufficient permissions.', 'aegisshield-security' ),
            );
        }

        if ( function_exists( 'is_multisite' ) && is_multisite() ) {
            return array(
                'success' => false,
                'message' => __( 'DB Prefix change is currently disabled on Multisite installations. (Safety block)', 'aegisshield-security' ),
            );
        }

        $this->plugin->get_logger()->log(
            'db_prefix_change_started',
            __( 'DB prefix change started.', 'aegisshield-security' ),
            'high',
            array(
                'old_prefix' => (string) $prefix_current,
                'new_prefix' => (string) $prefix_new,
                'mode'       => (string) $prefix_mode,
                'exec_mode'  => (string) $exec_mode,
                'backup'     => (string) $backup_file,
                'tables'     => is_array( $tables ) ? count( $tables ) : 0,
                'user_id'    => get_current_user_id(),
            )
        );

        $prefix_current = (string) $prefix_current;
        $prefix_new     = (string) $prefix_new;
        $prefix_mode    = (string) $prefix_mode;

        $prefix_current = $this->sanitize_db_prefix( $prefix_current );
        $prefix_new     = $this->sanitize_db_prefix( $prefix_new );

        if ( '' === $prefix_current || '' === $prefix_new ) {
            return array(
                'success' => false,
                'message' => __( 'Invalid database prefix supplied. Prefixes may only contain letters, numbers, and underscores.', 'aegisshield-security' ),
            );
        }

        if ( 'all' !== $prefix_mode ) {
            return array(
                'success' => false,
                'message' => __( 'Safety block: Apply is only allowed when Scope is set to "All tables". Preview-only is allowed for Core scope.', 'aegisshield-security' ),
            );
        }

        $validation_error = $this->validate_new_prefix_value( $prefix_new, $prefix_current );
        if ( $validation_error ) {
            return array(
                'success' => false,
                'message' => $validation_error,
            );
        }

        $backup_file = (string) $backup_file;
        if ( empty( $backup_file ) ) {
            return array(
                'success' => false,
                'message' => __( 'Backup selection is required before applying a prefix change.', 'aegisshield-security' ),
            );
        }

        $backup_dir = $this->get_db_tools_backup_dir();
        $backup_abs = trailingslashit( $backup_dir ) . basename( $backup_file );
        if ( empty( $backup_dir ) || ! file_exists( $backup_abs ) ) {
            return array(
                'success' => false,
                'message' => __( 'Selected backup file was not found. Create a new backup snapshot and try again.', 'aegisshield-security' ),
            );
        }

        $like = $wpdb->esc_like( $prefix_current ) . '%';
        $sql  = $wpdb->prepare( 'SHOW TABLES LIKE %s', $like );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $rows = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $like ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( empty( $rows ) ) {
            return array(
                'success' => false,
                'message' => __( 'No tables matched the current prefix. Aborting.', 'aegisshield-security' ),
            );
        }

        $core_suffixes = $this->get_wp_core_table_suffixes();
        $rename_map    = array();

        foreach ( $rows as $tbl ) {
            $tbl = (string) $tbl;

            // Only rename tables that truly start with current prefix.
            if ( 0 !== strpos( $tbl, $prefix_current ) ) {
                continue;
            }

            $suffix = substr( $tbl, strlen( $prefix_current ) );

            if ( 'core' === $prefix_mode ) {
                // Only core tables (single-site).
                if ( ! in_array( $suffix, $core_suffixes, true ) ) {
                    continue;
                }
            }

            $rename_map[ $tbl ] = $prefix_new . $suffix;
        }

        if ( empty( $rename_map ) ) {
            return array(
                'success' => false,
                'message' => __( 'No tables qualified for renaming under the selected scope.', 'aegisshield-security' ),
            );
        }

        // Pre-flight: ensure destination tables do not already exist.
        foreach ( $rename_map as $old => $new ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( ! empty( $exists ) ) {
                return array(
                    'success' => false,
                    'message' => sprintf(
                        /* translators: 1: destination table */
                        __( 'Destination table already exists: %s. Aborting to avoid collisions.', 'aegisshield-security' ),
                        esc_html( $new )
                    ),
                );
            }
        }

        $results = array(
            'renamed_tables' => array(),
            'options_rows'   => 0,
            'usermeta_rows'  => 0,
            'wpconfig'       => array( 'updated' => false, 'writable' => false, 'path' => '' ),
            'backup_used'    => basename( $backup_abs ),
            'prefix_current' => $prefix_current,
            'prefix_new'     => $prefix_new,
            'prefix_mode'    => $prefix_mode,
        );

        $dest_collisions = array();
        foreach ( $rename_map as $old => $new ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( ! empty( $exists ) ) {
                $dest_collisions[] = $new;
            }
        }

        if ( ! empty( $dest_collisions ) ) {
            return array(
                'success' => false,
                'message' => sprintf(
                    /* translators: %s: comma-separated table names. */
                    __( 'Safety block: one or more destination tables already exist. This usually means a previous migration partially completed. Existing destination tables: %s', 'aegisshield-security' ),
                    implode( ', ', array_slice( $dest_collisions, 0, 10 ) )
                ),
                'details' => array(
                    'collisions' => $dest_collisions,
                ),
            );
        }

        $renamed_new_to_old = array();

        foreach ( $rename_map as $old => $new ) {
            $old_s = $this->sanitize_db_identifier( $old );
            $new_s = $this->sanitize_db_identifier( $new );
            if ( '' === $old_s || '' === $new_s ) {
                return array(
                    'success' => false,
                    'message' => __( 'Invalid table name detected. Aborting.', 'aegisshield-security' ),
                );
            }

            $ok = $wpdb->query( $wpdb->prepare( 'RENAME TABLE %i TO %i', $old_s, $new_s ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.

            if ( false === $ok ) {
                // Rollback already renamed tables (best-effort).
                foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rn_s = $this->sanitize_db_identifier( $rolled_new );
                    $ro_s = $this->sanitize_db_identifier( $rolled_old );
                    if ( '' !== $rn_s && '' !== $ro_s ) {
                        $wpdb->query( $wpdb->prepare( 'RENAME TABLE %i TO %i', $rn_s, $ro_s ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                    }
                }

                return array(
                    'success' => false,
                    'message' => sprintf(
                        /* translators: 1: old table 2: new table */
                        __( 'Failed renaming table %1$s → %2$s. Rollback attempted. No further changes were applied.', 'aegisshield-security' ),
                        esc_html( $old ),
                        esc_html( $new )
                    ),
                    'details' => $results,
                );
            }

            $renamed_new_to_old[ $new ] = $old;
            $results['renamed_tables'][] = array( 'from' => $old, 'to' => $new );
        }


        $new_options  = $prefix_new . 'options';
        $new_usermeta = $prefix_new . 'usermeta';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $opt_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_options ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( ! empty( $opt_table_exists ) ) {
            $opt_like = $prefix_current . '%';
            $new_options_safe = $this->sanitize_db_identifier( $new_options );
            $opt_rows         = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                $wpdb->prepare(
                    "SELECT option_name FROM %i WHERE option_name LIKE %s",
                    $new_options_safe,
                    $opt_like
                )
            );

            if ( is_array( $opt_rows ) && ! empty( $opt_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $opt_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) ); // includes everything after old prefix
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE option_name = %s",
                            $new_options_safe,
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        // Collision: keep the existing target key, remove the old-prefixed one to avoid duplicates.
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete( $new_options, array( 'option_name' => $old_key ), array( '%s' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $new_options,
                        array( 'option_name' => $new_key ),
                        array( 'option_name' => $old_key ),
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        // Rollback already renamed tables (best-effort) before returning.
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rn_s = $this->sanitize_db_identifier( $rolled_new );
                    $ro_s = $this->sanitize_db_identifier( $rolled_old );
                    if ( '' !== $rn_s && '' !== $ro_s ) {
                        $wpdb->query( $wpdb->prepare( 'RENAME TABLE %i TO %i', $rn_s, $ro_s ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                    }
                        }
                        return array(
                            'success' => false,
                            'message' => __( 'Failed while updating options prefix-bound keys (collision-safe step).', 'aegisshield-security' ),
                        );
                    }

                    $updated++;
                }

                $results['options_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['options_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $um_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_usermeta ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( ! empty( $um_table_exists ) ) {
            $um_like = $prefix_current . '%';
            $new_usermeta_safe = $this->sanitize_db_identifier( $new_usermeta );
            $um_rows           = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                $wpdb->prepare(
                    "SELECT DISTINCT meta_key FROM %i WHERE meta_key LIKE %s",
                    $new_usermeta_safe,
                    $um_like
                )
            );

            if ( is_array( $um_rows ) && ! empty( $um_rows ) ) {
                $updated = 0;
                $skipped = 0;

                foreach ( $um_rows as $old_key ) {
                    $old_key = (string) $old_key;
                    $suffix  = substr( $old_key, strlen( $prefix_current ) );
                    $new_key = $prefix_new . $suffix;

                    if ( $new_key === $old_key ) {
                        continue;
                    }

                    $exists = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM %i WHERE meta_key = %s",
                            $new_usermeta_safe,
                            $new_key
                        )
                    );

                    if ( $exists > 0 ) {
                        // Collision: keep target key; delete old-prefixed rows to avoid duplicated capabilities keys.
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
                        $wpdb->delete( $new_usermeta, array( 'meta_key' => $old_key ), array( '%s' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $skipped++;
                        continue;
                    }

                    $ok = $wpdb->update( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $new_usermeta,
                        array( 'meta_key' => $new_key ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
                        array( 'meta_key' => $old_key ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
                        array( '%s' ),
                        array( '%s' )
                    );

                    if ( false === $ok ) {
                        foreach ( array_reverse( $renamed_new_to_old, true ) as $rolled_new => $rolled_old ) {
                    $rn_s = $this->sanitize_db_identifier( $rolled_new );
                    $ro_s = $this->sanitize_db_identifier( $rolled_old );
                    if ( '' !== $rn_s && '' !== $ro_s ) {
                        $wpdb->query( $wpdb->prepare( 'RENAME TABLE %i TO %i', $rn_s, $ro_s ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                    }
                        }
                        return array(
                            'success' => false,
                            'message' => __( 'Failed while updating usermeta prefix-bound keys (collision-safe step).', 'aegisshield-security' ),
                        );
                    }

                    $updated++;
                }

                $results['usermeta_rows'] = (int) $updated;
                if ( $skipped > 0 ) {
                    $results['usermeta_collisions_skipped'] = (int) $skipped;
                }
            }
        }

        update_option(
            'aegisshield_dbtools_last_prefix_migration',
            array(
                'old_prefix'  => $prefix_current,
                'new_prefix'  => $prefix_new,
                'timestamp'   => time(),
                'backup_file' => basename( $backup_abs ),
                'tables_renamed' => isset( $results['renamed_tables'] ) ? count( $results['renamed_tables'] ) : 0,
            ),
            false
        );

        $wpconfig_result = $this->try_update_wpconfig_table_prefix( $prefix_new );
        $results['wpconfig'] = $wpconfig_result;

        if ( empty( $wpconfig_result['updated'] ) ) {
            update_option( 'aegisshield_dbtools_pending_table_prefix', $prefix_new, false );
        } else {
            delete_option( 'aegisshield_dbtools_pending_table_prefix' );
        }

        $manual_line = '$table_prefix = \''
            . $prefix_new
            . '\';';

        $this->plugin->get_logger()->log(
            'db_prefix_change_completed',
            __( 'DB prefix change completed successfully.', 'aegisshield-security' ),
            'medium',
            array(
                'old_prefix' => (string) $prefix_current,
                'new_prefix' => (string) $prefix_new,
                'mode'       => (string) $prefix_mode,
                'exec_mode'  => (string) $exec_mode,
                'backup'     => isset( $results['backup_used'] ) ? $results['backup_used'] : (string) $backup_file,
                'details'    => $results,
                'user_id'    => get_current_user_id(),
            )
        );

return array(
            'success'         => true,
            'message'         => __( 'Prefix change applied. IMPORTANT: Ensure wp-config.php $table_prefix matches the NEW prefix before leaving this page.', 'aegisshield-security' ),
            'details'         => $results,
            'manual_wpconfig' => $manual_line,
        );
    }

    public function run_prefix_verification_scan( $expected_prefix, $old_prefix = '' ) {
        global $wpdb;

        $expected_prefix = (string) $expected_prefix;
        $old_prefix      = (string) $old_prefix;

        $issues   = array();
        $warnings = array();
        $counts   = array(
            'missing_core_tables'     => 0,
            'leftover_old_tables'     => 0,
            'leftover_old_option_keys'=> 0,
            'leftover_old_usermeta'   => 0,
        );
        $samples = array(
            'missing_core_tables' => array(),
            'leftover_old_tables' => array(),
            'leftover_old_option_keys' => array(),
            'leftover_old_usermeta' => array(),
        );

        // 1) Core table existence under expected prefix.
        $core_suffixes = array(
            'options',
            'users',
            'usermeta',
            'posts',
            'postmeta',
            'terms',
            'termmeta',
            'term_taxonomy',
            'term_relationships',
            'comments',
            'commentmeta',
            'links',
        );

        foreach ( $core_suffixes as $suffix ) {
            $t = $expected_prefix . $suffix;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $t ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( empty( $exists ) ) {
                $counts['missing_core_tables']++;
                $samples['missing_core_tables'][] = $t;
            }
        }

        if ( $counts['missing_core_tables'] > 0 ) {
            $issues[] = __( 'One or more required WordPress core tables are missing under the expected prefix. Do NOT proceed with any further changes until this is resolved.', 'aegisshield-security' );
        }

        if ( '' !== $old_prefix && $old_prefix !== $expected_prefix ) {
            $old_like = $wpdb->esc_like( $old_prefix ) . '%';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $leftover_tables = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', $old_like ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( is_array( $leftover_tables ) && ! empty( $leftover_tables ) ) {
                $counts['leftover_old_tables'] = count( $leftover_tables );
                $samples['leftover_old_tables'] = array_slice( $leftover_tables, 0, 25 );
                $warnings[] = __( 'Some tables still exist with the old prefix. This can be normal if you intentionally excluded certain plugin tables, but it becomes dangerous if wp-config.php has been updated to the new prefix while essential plugin tables remain on the old prefix.', 'aegisshield-security' );
            }

            $opt_table = $expected_prefix . 'options';
            $um_table  = $expected_prefix . 'usermeta';

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $opt_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $opt_table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( ! empty( $opt_exists ) ) {
                $opt_like = $old_prefix . '%';
                $opt_table_safe = $this->sanitize_db_identifier( $opt_table );
                $opt_count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE option_name LIKE %s",
                        $opt_table_safe,
                        $opt_like
                    )
                );
                $counts['leftover_old_option_keys'] = $opt_count;

                if ( $opt_count > 0 ) {
                    $rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $wpdb->prepare(
                            "SELECT option_name FROM %i WHERE option_name LIKE %s LIMIT 25",
                            $opt_table_safe,
                            $opt_like
                        )
                    );
                    $samples['leftover_old_option_keys'] = is_array( $rows ) ? $rows : array();
                    $warnings[] = __( 'Some option_name values still begin with the old prefix. This usually includes keys like {prefix}user_roles and should be updated to match the new prefix.', 'aegisshield-security' );
                }
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
            $um_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $um_table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
            if ( ! empty( $um_exists ) ) {
                $um_like = $old_prefix . '%';
                $um_table_safe = $this->sanitize_db_identifier( $um_table );
                $um_count = (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM %i WHERE meta_key LIKE %s",
                        $um_table_safe,
                        $um_like
                    )
                );
                $counts['leftover_old_usermeta'] = $um_count;

                if ( $um_count > 0 ) {
                    $rows = $wpdb->get_col( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
                        $wpdb->prepare(
                            "SELECT DISTINCT meta_key FROM %i WHERE meta_key LIKE %s LIMIT 25",
                            $um_table_safe,
                            $um_like
                        )
                    );
                    $samples['leftover_old_usermeta'] = is_array( $rows ) ? $rows : array();
                    $warnings[] = __( 'Some usermeta meta_key values still begin with the old prefix (example: wp_capabilities). These must be updated to the new prefix or user roles/capabilities may break.', 'aegisshield-security' );
                }
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $as_any = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', '%actionscheduler_actions' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( is_array( $as_any ) && ! empty( $as_any ) ) {
            $expected_as = $expected_prefix . 'actionscheduler_actions';
            $expected_present = in_array( $expected_as, $as_any, true );
            if ( ! $expected_present ) {
                $warnings[] = __( 'Action Scheduler tables were detected, but not under the expected prefix. If wp-config.php is set to the expected prefix, Action Scheduler-based cron jobs may error until those tables are aligned.', 'aegisshield-security' );
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
        $aegis_any = $wpdb->get_col( $wpdb->prepare( 'SHOW TABLES LIKE %s', '%aegisshield_activity_log' ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct DB access is required for custom tables; caching handled where appropriate.
        if ( is_array( $aegis_any ) && ! empty( $aegis_any ) ) {
            $expected_aegis = $expected_prefix . 'aegisshield_activity_log';
            $expected_present = in_array( $expected_aegis, $aegis_any, true );
            if ( ! $expected_present ) {
                $warnings[] = __( 'AegisShield custom tables were detected, but not under the expected prefix. This can cause AegisShield cron maintenance (log pruning, etc.) to error until those tables are aligned.', 'aegisshield-security' );
            }
        }

        return array(
            'success'  => empty( $issues ),
            'issues'   => $issues,
            'warnings' => $warnings,
            'counts'   => $counts,
            'samples'  => $samples,
        );
    }


    private function validate_new_prefix_value( $new_prefix, $current_prefix ) {
        $new_prefix     = (string) $new_prefix;
        $current_prefix = (string) $current_prefix;

        if ( empty( $new_prefix ) ) {
            return __( 'New prefix is required.', 'aegisshield-security' );
        }

        // Only letters, numbers, underscores. Must end with underscore.
        if ( ! preg_match( '/^[A-Za-z0-9_]+$/', $new_prefix ) ) {
            return __( 'Prefix contains invalid characters. Use only letters, numbers, and underscores.', 'aegisshield-security' );
        }

        if ( '_' !== substr( $new_prefix, -1 ) ) {
            return __( 'Prefix must end with an underscore ( _ ).', 'aegisshield-security' );
        }

        if ( $new_prefix === $current_prefix ) {
            return __( 'New prefix must be different from the current prefix.', 'aegisshield-security' );
        }

        if ( strlen( $new_prefix ) > 20 ) {
            return __( 'Prefix is too long. Keep it under 20 characters.', 'aegisshield-security' );
        }

        return '';
    }

    private function get_wp_core_table_suffixes() {
        return array(
            'commentmeta',
            'comments',
            'links',
            'options',
            'postmeta',
            'posts',
            'termmeta',
            'terms',
            'term_relationships',
            'term_taxonomy',
            'usermeta',
            'users',
        );
    }

private function try_update_wpconfig_table_prefix( $new_prefix ) {
    $path = defined( 'ABSPATH' ) ? trailingslashit( ABSPATH ) . 'wp-config.php' : '';
    $fs   = $this->get_filesystem();

    $writable = ( $fs && ! empty( $path ) && file_exists( $path ) && $fs->is_writable( $path ) );

    if ( ! $writable && defined( 'ABSPATH' ) ) {
        $alt = dirname( ABSPATH ) . '/wp-config.php';
        if ( file_exists( $alt ) ) {
            $path     = $alt;
            $writable = ( $fs && $fs->is_writable( $path ) );
        }
    }

    $result = array(
        'updated'  => false,
        'writable' => (bool) $writable,
        'path'     => (string) $path,
    );

    if ( ! $writable || empty( $path ) || ! $fs ) {
        return $result;
    }

    $contents = $fs->get_contents( $path );
    if ( false === $contents || empty( $contents ) ) {
        return $result;
    }

    $pattern = '/\$table_prefix\s*=\s*[\'"][A-Za-z0-9_]*[\'"]\s*;/';

    if ( ! preg_match( $pattern, $contents ) ) {
        return $result;
    }

    $replacement = '$table_prefix = \''
        . $new_prefix
        . '\';';

    $new_contents = preg_replace( $pattern, $replacement, $contents, 1 );
    if ( empty( $new_contents ) || $new_contents === $contents ) {
        return $result;
    }

    $written = $fs->put_contents( $path, $new_contents, FS_CHMOD_FILE );
    if ( ! $written ) {
        return $result;
    }

    $result['updated'] = true;
    return $result;
}


}