<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Module_Hardening implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'hardening';
    }

    public function register_settings() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';
        $defaults = [
            // Master toggle: Hardening is completely OFF unless explicitly enabled by the admin.
            'enable_hardening'           => 'off',
            'disable_file_edit'          => 'off',
            'disable_xmlrpc_behavior'    => 'c',
            'block_user_enum_behavior'   => 'a',
            'hide_wp_version'            => 'off',
            'disable_editor_screens'     => 'off',
            'enforce_strong_passwords'   => 'off',
            'password_min_length'        => 12,
            'password_require_uppercase' => 'on',
            'password_require_lowercase' => 'on',
            'password_require_number'    => 'on',
            'password_require_symbol'    => 'on',
            'hardening_vuln_source'      => 'local',
            'hardening_vuln_patchstack_key' => '',
            'hardening_vuln_last_run'    => 0,
            'hardening_vuln_results'     => [],
            'hardening_role_risk_mode'   => 'off',
            'hardening_role_risk_results'=> [],
            'hardening_rest_mode'        => 'default',
            'hardening_xmlrpc_mode'      => 'default',
            'hardening_audit_enabled'    => 'off',
            'hardening_audit_last_sent'  => 0,
        ];

        foreach ( $defaults as $key => $value ) {
            if ( null === $settings->get( $section, $key, null ) ) {
                $settings->set( $section, $key, $value );
            }
        }

        $settings->save();
    }

    public function init() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        // WordPress-friendly safety: do not hook/alter core behavior unless the admin explicitly enables Hardening.
        $enabled = (string) $settings->get( $section, 'enable_hardening', 'off' );
        if ( ! in_array( $enabled, [ 'on', '1', 'yes' ], true ) ) {
            return;
        }

        $disable_file_edit      = (string) $settings->get( $section, 'disable_file_edit', 'off' );
        $xmlrpc_behavior        = (string) $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );
        $block_user_enum        = (string) $settings->get( $section, 'block_user_enum_behavior', 'a' );
        $hide_wp_version        = (string) $settings->get( $section, 'hide_wp_version', 'off' );
        $disable_editor_screens = (string) $settings->get( $section, 'disable_editor_screens', 'off' );
        $enforce_passwords      = (string) $settings->get( $section, 'enforce_strong_passwords', 'off' );

        if ( in_array( $disable_file_edit, [ 'on', '1', 'yes' ], true ) ) {
            if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
                define( 'DISALLOW_FILE_EDIT', true );
            }
            add_filter( 'editable_extensions', [ $this, 'disable_editable_extensions' ] );
        }

		add_action( 'load-theme-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'theme-editor' ]
				);
			}
		});

		add_action( 'load-plugin-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'plugin-editor' ]
				);
			}
		});

        /* XML-RPC handling */
        if ( $xmlrpc_behavior === 'c' ) {
            add_filter( 'xmlrpc_methods', function ( $methods ) {
                unset( $methods['wp.deletePage'], $methods['wp.deletePost'] );
                unset( $methods['wp.editPage'], $methods['wp.editPost'] );
                unset( $methods['wp.newPage'], $methods['wp.newPost'] );
                return $methods;
            });
        } else {
            add_filter( 'xmlrpc_enabled', '__return_false' );
            add_action( 'init', [ $this, 'handle_xmlrpc_request' ] );
        }

        /* Block user enumeration */
        add_action( 'template_redirect', [ $this, 'block_user_enumeration' ] );

        /* Hide WP version */
        if ( $hide_wp_version === 'on' ) {
            add_action( 'init', [ $this, 'hide_wp_version_meta' ] );
            add_filter( 'the_generator', '__return_empty_string' );
            add_filter( 'style_loader_src', [ $this, 'strip_version_from_assets' ], 9999 );
            add_filter( 'script_loader_src', [ $this, 'strip_version_from_assets' ], 9999 );
        }

        /* Disable editor screens */
        if ( $disable_editor_screens === 'on' ) {
            add_action( 'admin_menu', [ $this, 'remove_editor_submenus' ], 999 );
        }
		
		add_action( 'load-theme-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'theme-editor' ]
				);
			}
		});

		add_action( 'load-plugin-editor.php', function() use ( $disable_file_edit ) {
			if ( $disable_file_edit === 'on' ) {
				$this->log_hardening_violation(
					'disable_file_edit',
					'high',
					[ 'screen' => 'plugin-editor' ]
				);
			}
		});

        /* Enforce strong passwords */
        if ( $enforce_passwords === 'on' ) {
            // Add plugin nonces to core wp-login.php forms so POST validation can verify intent (CSRF protection).
            add_action( 'register_form', [ $this, 'render_register_password_nonce' ] );
            add_action( 'resetpass_form', [ $this, 'render_reset_password_nonce' ] );
            add_action( 'user_profile_update_errors', [ $this, 'enforce_password_strength_profile' ], 10, 3 );
            add_action( 'register_post', [ $this, 'enforce_password_strength_registration' ], 10, 3 );
            add_action( 'validate_password_reset', [ $this, 'enforce_password_strength_reset' ], 10, 2 );
        }

        $this->init_rest_api_controls();
        $this->init_xmlrpc_controls();
        $this->init_weekly_hardening_audit();

    }


    protected function init_rest_api_controls() {
        $settings = $this->plugin->get_settings();
        $mode = $settings->get( 'hardening', 'hardening_rest_mode', 'default' );

        add_filter( 'rest_authentication_errors', function( $result ) use ( $mode ) {

            if ( is_user_logged_in() ) { return $result; }
            if ( $mode === 'default' ) { return $result; }

            $route = '';

            if ( isset( $_SERVER['REQUEST_URI'] ) ) {
                $route = sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) );
            }

            if ( $mode === 'safe' ) {
                if ( strpos( $route, '/wp/v2/users' ) !== false ) {
                    $this->log_hardening_violation(
						'hardening_rest_mode',
						'high',
						[ 'route' => $route ]
					);

					return new \WP_Error( 'rest_blocked', 'REST endpoint blocked for security.', [ 'status' => 403 ] );
                }
                return $result;
            }

            if ( $mode === 'hardened' ) {
                $allowed = [ 'wp/v2/posts', 'wp/v2/categories', 'wp/v2/tags' ];
                foreach ( $allowed as $a ) {
                    if ( strpos( $route, $a ) !== false ) {
                        return $result;
                    }
                }
                return new \WP_Error( 'rest_blocked', 'Anonymous REST API blocked.', [ 'status' => 403 ] );
            }

            return $result;
        });
    }

    protected function init_xmlrpc_controls() {
        $settings = $this->plugin->get_settings();
        $mode = $settings->get( 'hardening', 'hardening_xmlrpc_mode', 'default' );

        if ( $mode === 'disabled' ) {
            add_filter( 'xmlrpc_enabled', '__return_false' );
            return;
        }

        if ( $mode === 'jetpack' ) {
            add_filter( 'xmlrpc_methods', function( $methods ) {
                $allowed = [
                    'system.multicall',
                    'system.getCapabilities',
                    'jetpack.verify'
                ];
                return array_intersect_key( $methods, array_flip( $allowed ) );
            });
        }
    }

    protected function init_weekly_hardening_audit() {

        if ( ! wp_next_scheduled( 'aegisshield_hardening_audit_weekly' ) ) {
            wp_schedule_event( time() + 3600, 'weekly', 'aegisshield_hardening_audit_weekly' );
        }

        add_action( 'aegisshield_hardening_audit_weekly', function() {

            $settings = $this->plugin->get_settings();
            $section  = 'hardening';

            if ( $settings->get( $section, 'hardening_audit_enabled', 'off' ) !== 'on' ) {
                return;
            }

            $vuln = $settings->get( $section, 'hardening_vuln_results', [] );
            $roles = $settings->get( $section, 'hardening_role_risk_results', [] );

            $body = "AegisShield Weekly Hardening Report\n\n";
            $body .= "Vulnerability Findings:\n" . wp_json_encode( $vuln, JSON_PRETTY_PRINT ) . "\n\n";
            $body .= "Role & Capability Risk Findings:\n" . wp_json_encode( $roles, JSON_PRETTY_PRINT ) . "\n\n";

            $notifier = $this->plugin->get_notifier();
            $notifier->send_admin_alert( 'Weekly Hardening Report', nl2br( $body ) );

            $settings->set( $section, 'hardening_audit_last_sent', time() );
            $settings->save();
        });
    }

    public function run_vulnerability_scan() {

        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        $source  = $settings->get( $section, 'hardening_vuln_source', 'local' );
        $api_key = $settings->get( $section, 'hardening_vuln_patchstack_key', '' );

        $results = [];

        /* Plugins */
        foreach ( get_plugins() as $slug => $data ) {
            $version = $data['Version'] ?? '';
            $risk = $this->lookup_plugin_vulnerability( $source, $slug, $version, $api_key );
            if ( $risk ) {
                $results['plugins'][$slug] = $risk;
            }
        }

        /* Themes */
        foreach ( wp_get_themes() as $slug => $theme ) {
            $version = $theme->get( 'Version' );
            $risk = $this->lookup_theme_vulnerability( $source, $slug, $version, $api_key );
            if ( $risk ) {
                $results['themes'][$slug] = $risk;
            }
        }

        $settings->set( $section, 'hardening_vuln_results', $results );
        $settings->set( $section, 'hardening_vuln_last_run', time() );
        $settings->save();

        return $results;
    }

    public function run_role_risk_scan() {

        $dangerous_caps = [
            'manage_options',
            'edit_plugins',
            'delete_users',
            'unfiltered_html',
            'edit_theme_options'
        ];

        $results = [];

        foreach ( get_users() as $u ) {

            $caps = array_keys( $u->allcaps );
            $hit  = array_intersect( $caps, $dangerous_caps );

            $last_login = get_user_meta( $u->ID, 'last_login', true );
            if ( ! $last_login ) {
                $last_login = strtotime( $u->user_registered );
            }

            $inactive = floor( ( time() - $last_login ) / DAY_IN_SECONDS );

            $risk = 'low';
            if ( ! empty( $hit ) ) { $risk = 'medium'; }
            if ( $inactive > 30 && ! empty( $hit ) ) { $risk = 'high'; }

            if ( $risk !== 'low' ) {
                $results[] = [
                    'user_id'       => $u->ID,
                    'username'      => $u->user_login,
                    'role'          => implode( ',', $u->roles ),
                    'dangerous_caps'=> $hit,
                    'last_login'    => $last_login,
                    'risk'          => $risk
                ];
            }
        }

        $settings = $this->plugin->get_settings();
        $settings->set( 'hardening', 'hardening_role_risk_results', $results );
        $settings->save();

        return $results;
    }

    protected function lookup_plugin_vulnerability( $source, $slug, $version, $api_key ) {

        if ( $source === 'local' ) {
            return $this->local_risk_heuristic( $version );
        }

        if ( $source === 'patchstack' ) {
            return $this->patchstack_lookup( $slug, $version, $api_key );
        }

        if ( $source === 'wpvulndb' ) {
            return $this->wpvulndb_lookup( $slug, $version );
        }

        return null;
    }

    protected function lookup_theme_vulnerability( $source, $slug, $version, $api_key ) {
        return $this->lookup_plugin_vulnerability( $source, $slug, $version, $api_key );
    }

    protected function local_risk_heuristic( $version ) {
        if ( empty( $version ) ) {
            return [
                'risk'   => 'medium',
                'reason' => 'Unknown version, cannot determine safety.'
            ];
        }

        return [
            'risk'   => 'low',
            'reason' => 'No external vulnerability feed enabled.'
        ];
    }

    protected function patchstack_lookup( $slug, $version, $api_key ) {

        if ( empty( $api_key ) ) {
            return null;
        }

        $url  = "https://api.patchstack.com/v1/plugin/" . urlencode( $slug );
        $args = [
            'headers' => [ 'Authorization' => 'Bearer ' . $api_key ],
            'timeout' => 6,
        ];

        $res = wp_remote_get( $url, $args );
        if ( is_wp_error( $res ) ) { return null; }

        $body = json_decode( wp_remote_retrieve_body( $res ), true );
        if ( ! $body || empty( $body['vulnerabilities'] ) ) { return null; }

        return [
            'risk'   => 'high',
            'reason' => 'Patchstack reports known vulnerabilities.',
            'raw'    => $body['vulnerabilities']
        ];
    }

    protected function wpvulndb_lookup( $slug, $version ) {

        $url  = "https://wpvulndb.com/api/v3/plugins/" . urlencode( $slug );
        $args = [ 'timeout' => 6 ];

        $res = wp_remote_get( $url, $args );
        if ( is_wp_error( $res ) ) { return null; }

        $body = json_decode( wp_remote_retrieve_body( $res ), true );
        if ( ! $body || empty( $body[$slug]['vulnerabilities'] ) ) { return null; }

        return [
            'risk'   => 'high',
            'reason' => 'WPVulnDB reports known vulnerabilities.',
            'raw'    => $body[$slug]['vulnerabilities']
        ];
    }

	protected function log_hardening_violation( $feature_key, $severity = 'medium', array $context = [] ) {

		if ( ! method_exists( $this->plugin, 'get_logger' ) ) {
			return;
		}

		$logger = $this->plugin->get_logger();
		if ( ! $logger ) {
			return;
		}

		$user_id = get_current_user_id();

		$payload = array_merge(
			[
				'feature_key'  => $feature_key,
				'user_id'      => $user_id ?: 0,
				'requested_url'=> ( isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '' ),
				'ip_address'   => ( isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '' ),
			],
			$context
		);

		$logger->log(
			'hardening_violation',
			$severity,
			$payload
		);
	}

    public function disable_editable_extensions( $extensions ) {
        if ( isset( $extensions['php'] ) ) {
            unset( $extensions['php'] );
        }
        return $extensions;
    }

    public function handle_xmlrpc_request() {
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';
        $xmlrpc_behavior = $settings->get( $section, 'disable_xmlrpc_behavior', 'c' );

		if ( $xmlrpc_behavior === 'a' ) {

			$this->log_hardening_violation(
				'disable_xmlrpc_behavior',
				'high',
				[ 'method' => 'xmlrpc' ]
			);

			wp_die( 'XML-RPC requests have been disabled by AegisShield.', '', [ 'response' => 403 ] );
        } elseif ( $xmlrpc_behavior === 'b' ) {

			$this->log_hardening_violation(
				'disable_xmlrpc_behavior',
				'high',
				[ 'method' => 'xmlrpc' ]
			);

			status_header( 403 );
			exit;
		}
    }

    public function block_user_enumeration() {
        if ( ! is_admin() && isset( $_GET['author'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No state change; blocks user enumeration via URL param.
            $settings = $this->plugin->get_settings();
            $behavior = $settings->get( 'hardening', 'block_user_enum_behavior', 'a' );

			if ( $behavior === 'a' ) {

				$this->log_hardening_violation(
					'block_user_enum_behavior',
					'medium',
					[ 'query' => ( isset( $_SERVER['QUERY_STRING'] ) ? sanitize_text_field( wp_unslash( $_SERVER['QUERY_STRING'] ) ) : '' ) ]
				);

				wp_safe_redirect( home_url( '/' ) );
				exit;
            } else {
                global $wp_query;
				$this->log_hardening_violation(
					'block_user_enum_behavior',
					'medium',
					[ 'query' => ( isset( $_SERVER['QUERY_STRING'] ) ? sanitize_text_field( wp_unslash( $_SERVER['QUERY_STRING'] ) ) : '' ) ]
				);

				$wp_query->set_404();
				status_header( 404 );
				nocache_headers();
            }
        }
    }

    public function hide_wp_version_meta() {
        remove_action( 'wp_head', 'wp_generator' );
    }

    public function strip_version_from_assets( $src ) {
        if ( strpos( $src, 'ver=' ) !== false ) {
            $src = remove_query_arg( 'ver', $src );
        }
        return $src;
    }

    public function remove_editor_submenus() {
        remove_submenu_page( 'themes.php', 'theme-editor.php' );
        remove_submenu_page( 'plugins.php', 'plugin-editor.php' );
    }

    /**
     * Output a nonce field on the core wp-login.php registration form.
     *
     * This allows our server-side registration validation (register_post) to
     * verify intent and satisfy WordPress nonce best practices.
     */
    public function render_register_password_nonce() {
        wp_nonce_field( 'as_hardening_register_password', 'as_hardening_register_password_nonce' );
    }

    /**
     * Output a nonce field on the core wp-login.php reset password form.
     *
     * This allows our server-side reset validation (validate_password_reset) to
     * verify intent and satisfy WordPress nonce best practices.
     */
    public function render_reset_password_nonce() {
        wp_nonce_field( 'as_hardening_reset_password', 'as_hardening_reset_password_nonce' );
    }

    public function enforce_password_strength_profile( $errors, $update, $user ) {
        // WordPress core profile update uses an update-user_{ID} nonce. Verify when present to satisfy security sniffing.
        if ( isset( $_POST['_wpnonce'] ) ) {
            $nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );
            if ( ! wp_verify_nonce( $nonce, 'update-user_' . $user->ID ) ) {
                return;
            }
        }

        if ( ! empty( $_POST['pass1'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be sanitized; they are validated but must remain unchanged for strength checks.
            $password = (string) wp_unslash( $_POST['pass1'] );
            $msg      = $this->validate_password_strength( $password, $user->ID );
            if ( $msg ) { $errors->add( 'weak_password', $msg ); }
        }
    }

    public function enforce_password_strength_registration( $sanitized_user_login, $user_email, $errors ) {
        $password = '';

        // Verify our registration nonce added via register_form().
        // If the nonce is missing/invalid, do not process posted passwords.
        if ( empty( $_POST['as_hardening_register_password_nonce'] ) ) {
            return $errors;
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['as_hardening_register_password_nonce'] ) );
        if ( ! wp_verify_nonce( $nonce, 'as_hardening_register_password' ) ) {
            return $errors;
        }

        // Support common password field names (core + popular forms).
        if ( ! empty( $_POST['pass1'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be sanitized; they are validated but must remain unchanged for strength checks.
            $password = (string) wp_unslash( $_POST['pass1'] );
        } elseif ( ! empty( $_POST['user_pass'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be sanitized; they are validated but must remain unchanged for strength checks.
            $password = (string) wp_unslash( $_POST['user_pass'] );
        } elseif ( ! empty( $_POST['password'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be sanitized; they are validated but must remain unchanged for strength checks.
            $password = (string) wp_unslash( $_POST['password'] );
        }

        if ( $password !== '' ) {
            $msg = $this->validate_password_strength( $password, 0 );
            if ( $msg ) {
                $errors->add( 'weak_password', $msg );
            }
        }

        return $errors;
    }

    public function enforce_password_strength_reset( $errors, $user ) {
        // Verify our reset nonce added via resetpass_form().
        // If the nonce is missing/invalid, do not process posted passwords.
        if ( empty( $_POST['as_hardening_reset_password_nonce'] ) ) {
            return;
        }

        $nonce = sanitize_text_field( wp_unslash( $_POST['as_hardening_reset_password_nonce'] ) );
        if ( ! wp_verify_nonce( $nonce, 'as_hardening_reset_password' ) ) {
            return;
        }

        if ( ! empty( $_POST['pass1'] ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Passwords should not be sanitized; they are validated but must remain unchanged for strength checks.
            $password = (string) wp_unslash( $_POST['pass1'] );
            $msg = $this->validate_password_strength( $password, $user->ID );
            if ( $msg ) { $errors->add( 'weak_password', $msg ); }
        }
    }

    protected function validate_password_strength( $password, $user_id ) {
        $password = (string) $password;
        $settings = $this->plugin->get_settings();
        $section  = 'hardening';

        $min  = (int) $settings->get( $section, 'password_min_length', 12 );
        $up   = $settings->get( $section, 'password_require_uppercase', 'on' );
        $low  = $settings->get( $section, 'password_require_lowercase', 'on' );
        $num  = $settings->get( $section, 'password_require_number', 'on' );
        $sym  = $settings->get( $section, 'password_require_symbol', 'on' );

		if ( strlen( $password ) < $min ) {
			$this->log_hardening_violation(
				'enforce_strong_passwords',
				'medium',
				[ 'reason' => 'min_length' ]
			);
			return "Password must be at least {$min} characters.";
		}
        if ( $up === 'on'  && ! preg_match( '/[A-Z]/', $password ) ) return "Password must contain uppercase letters.";
        if ( $low === 'on' && ! preg_match( '/[a-z]/', $password ) ) return "Password must contain lowercase letters.";
        if ( $num === 'on' && ! preg_match( '/[0-9]/', $password ) ) return "Password must contain numbers.";
        if ( $sym === 'on' && ! preg_match( '/[^a-zA-Z0-9]/', $password ) ) return "Password must contain symbols.";

        return '';
    }
}