<?php
namespace AegisShield\Modules;

use AegisShield\AS_Plugin;

defined( 'ABSPATH' ) || exit;

class AS_Module_Sec_Headers implements AS_Module_Interface {

    protected $plugin;

    public function __construct( AS_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function get_slug() {
        return 'sec_headers';
    }

    public function register_settings() {
    }

    public function init() {
        add_action( 'send_headers', array( $this, 'send_security_headers' ) );
        add_action( 'init', array( $this, 'maybe_handle_csp_report' ) );
    }


    protected function is_pro_active() {
        return function_exists( 'aegisshield_is_pro_active' ) && aegisshield_is_pro_active();
    }

	protected function log_violation( $code, $message, array $ctx = array(), $event = 'sec_headers_violation' ) {
		$ctx['module'] = 'sec_headers';
		$ctx['code']   = (string) $code;

		// Add safe request metadata (non-sensitive).
		$ctx['is_ssl'] = function_exists( 'is_ssl' ) ? ( is_ssl() ? 1 : 0 ) : 0;
		$ctx['uri']    = isset( $_SERVER['REQUEST_URI'] ) ? substr( sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ), 0, 500 ) : '';
		$ctx['method'] = isset( $_SERVER['REQUEST_METHOD'] ) ? substr( sanitize_key( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ), 0, 20 ) : '';

		$logger = $this->plugin ? $this->plugin->get_logger() : null;
		if ( $logger ) {
			$logger->log( $event, $message, $ctx );
			return;
		}

	}

	public function send_security_headers() {
		if ( headers_sent() ) {
			$this->log_violation(
				'headers_already_sent',
				__( 'Security Headers enabled, but headers were already sent so AegisShield could not apply them.', 'aegisshield-security' ),
				array()
			);
			return;
		}

        $settings = $this->plugin->get_settings();
        $section  = 'sec_headers';

        $enabled = (string) $settings->get( $section, 'enabled', '' );
        if ( 'on' !== $enabled ) {
            return;
        }

        $existing = function_exists( 'headers_list' ) ? headers_list() : array();

        $has = function( $name ) use ( $existing ) {
            foreach ( $existing as $header_line ) {
                if ( stripos( $header_line, $name . ':' ) === 0 ) {
                    return true;
                }
            }
            return false;
        };

        // Determine which logical area this request belongs to.
        $area = $this->get_request_area( $settings, $section );

        // Standard headers – same for all areas for now.
        if ( ! $has( 'X-Frame-Options' ) ) {
            header( 'X-Frame-Options: SAMEORIGIN' );
        }
        if ( ! $has( 'X-Content-Type-Options' ) ) {
            header( 'X-Content-Type-Options: nosniff' );
        }
        if ( ! $has( 'Referrer-Policy' ) ) {
            header( 'Referrer-Policy: strict-origin-when-cross-origin' );
        }
        if ( ! $has( 'X-XSS-Protection' ) ) {
            header( 'X-XSS-Protection: 1; mode=block' );
        }

        // HSTS.
        $enable_hsts = (string) $settings->get( $section, 'enable_hsts', '' );
        if ( 'on' === $enable_hsts && is_ssl() && ! $has( 'Strict-Transport-Security' ) ) {
            header( 'Strict-Transport-Security: max-age=31536000; includeSubDomains' );
        }
		
		if ( 'on' === $enable_hsts && ! is_ssl() ) {
			$this->log_violation(
				'hsts_not_https',
				__( 'HSTS is enabled, but the current request is not HTTPS so HSTS cannot be applied.', 'aegisshield-security' ),
				array()
			);
		}

        // CSP – supports presets and Visual CSP Builder (Pro).
        $csp_level = (string) $settings->get( $section, 'csp_level', '' );

        // Per-area overrides (Pro feature).
        if ( $this->is_pro_active() ) {
            $override = '';
            if ( 'frontend' === $area ) {
                $override = (string) $settings->get( $section, 'csp_level_frontend', '' );
            } elseif ( 'admin' === $area ) {
                $override = (string) $settings->get( $section, 'csp_level_admin', '' );
            } elseif ( 'custom' === $area ) {
                $override = (string) $settings->get( $section, 'csp_level_custom', '' );
            }

            if ( '' !== $override && 'inherit' !== $override ) {
                $csp_level = $override;
            }
        }

        $builder_mode = (string) $settings->get( $section, 'builder_mode', 'off' );
        $use_builder  = in_array( $csp_level, array( 'builder_enforce', 'builder_report' ), true ) && 'off' !== $builder_mode && $this->is_pro_active();

        $has_csp        = $has( 'Content-Security-Policy' );
        $has_csp_report = $has( 'Content-Security-Policy-Report-Only' );

		if ( $use_builder ) {
			$policies = $this->build_csp_from_builder( $settings, $section );

			if ( empty( $policies['enforced'] ) && empty( $policies['report'] ) ) {
				$this->log_violation(
					'csp_builder_empty',
					__( 'Visual CSP Builder is selected, but the generated policy is empty. Nothing was applied.', 'aegisshield-security' ),
					array(
						'area'        => $area,
						'csp_level'    => $csp_level,
						'builder_mode' => (string) $settings->get( $section, 'builder_mode', 'off' ),
					)
				);
			} else {
				$builder_mode    = (string) $settings->get( $section, 'builder_mode', 'off' );
				$has_report_only = ( 'builder_report' === $csp_level || 'report_only' === $builder_mode );

				$endpoint = $this->get_csp_report_uri();

				// ENFORCED policy
				if ( ! empty( $policies['enforced'] ) && ! $has_csp ) {
					$enforced_policy = $policies['enforced'];
					if ( $endpoint ) {
						$enforced_policy .= '; report-uri ' . $endpoint;
					}
					header( 'Content-Security-Policy: ' . $enforced_policy );
				}

				// REPORT-ONLY policy
				if ( ! empty( $policies['report'] ) && ! $has_csp_report && $has_report_only ) {
					$report_policy = $policies['report'];
					if ( $endpoint ) {
						$report_policy .= '; report-uri ' . $endpoint;
					}
					header( 'Content-Security-Policy-Report-Only: ' . $report_policy );
				}

				// If builder is set to report-only but enforced CSP already exists elsewhere, log that we could not set ours.
				if ( $has_report_only && $has_csp_report ) {
					$this->log_violation(
						'csp_report_only_header_exists',
						__( 'CSP Builder is in Report-Only mode, but Content-Security-Policy-Report-Only header already exists so AegisShield did not override it.', 'aegisshield-security' ),
						array(
							'area'        => $area,
							'csp_level'    => $csp_level,
							'builder_mode' => $builder_mode,
						)
					);
				}

				if ( ! $has_report_only && $has_csp ) {
					$this->log_violation(
						'csp_enforce_header_exists',
						__( 'CSP Builder is in Enforced mode, but Content-Security-Policy header already exists so AegisShield did not override it.', 'aegisshield-security' ),
						array(
							'area'        => $area,
							'csp_level'    => $csp_level,
							'builder_mode' => $builder_mode,
						)
					);
				}
			}
		} else {

            $report_only_requested = ( 'report_only' === $builder_mode );

            if ( $report_only_requested ) {
                if ( ! $has_csp_report ) {
                    $csp = $this->build_csp( $csp_level );
                    if ( $csp ) {
                        $endpoint = $this->get_csp_report_uri();
                        if ( $endpoint ) {
                            $csp .= '; report-uri ' . $endpoint;
                        }
                        header( 'Content-Security-Policy-Report-Only: ' . $csp );
                    }
                }
            } else {
                if ( ! $has_csp ) {
                    $csp = $this->build_csp( $csp_level );
                    if ( $csp ) {
                        header( 'Content-Security-Policy: ' . $csp );
                    }
                }
            }
        }
    }

    protected function build_csp( $lvl ) {
        $lvl = strtolower( (string) $lvl );
        if ( '' === $lvl || 'none' === $lvl ) {
            return '';
        }

        if ( 'strict' === $lvl ) {
            return implode(
                '; ',
                array(
                    "default-src 'self'",
                    "img-src 'self' data:",
                    "script-src 'self'",
                    "style-src 'self' 'unsafe-inline'",
                    "font-src 'self' data:",
                    "connect-src 'self'",
                    "frame-ancestors 'self'",
                )
            );
        }

        return implode(
            '; ',
            array(
                "default-src 'self'",
                "img-src 'self' data: https:",
                "script-src 'self' 'unsafe-inline' 'unsafe-eval' https:",
                "style-src 'self' 'unsafe-inline' https:",
                "font-src 'self' data: https:",
                "connect-src 'self' https:",
                "frame-ancestors 'self'",
            )
        );
    }

    protected function build_csp_from_builder( $settings, $section ) {
		$mode_script  = (string) $settings->get( $section, 'builder_script_mode', 'off' );
		$mode_style   = (string) $settings->get( $section, 'builder_style_mode', 'off' );
		$mode_img     = (string) $settings->get( $section, 'builder_img_mode', 'off' );
		$mode_font    = (string) $settings->get( $section, 'builder_font_mode', 'off' );
		$mode_connect = (string) $settings->get( $section, 'builder_connect_mode', 'off' );
		$mode_frame   = (string) $settings->get( $section, 'builder_frame_mode', 'off' );
		$mode_extra   = (string) $settings->get( $section, 'builder_extra_mode', 'off' );
        // Script sources.
        $script = array( "'self'" );
        $script_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_script_src', '' )
        );
        $script = array_merge( $script, $script_extra );
        if ( 'on' === (string) $settings->get( $section, 'builder_allow_inline_js', '' ) ) {
            $script[] = "'unsafe-inline'";
        }

        // Style sources.
        $style = array( "'self'" );
        $style_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_style_src', '' )
        );
        $style = array_merge( $style, $style_extra );
        if ( 'on' === (string) $settings->get( $section, 'builder_allow_inline_css', '' ) ) {
            $style[] = "'unsafe-inline'";
        }

        // Image sources.
        $img = array( "'self'", 'data:' );
        $img_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_img_src', '' )
        );$directives[] = "default-src 'self'";
        $img = array_merge( $img, $img_extra );

        // Font sources.
        $font = array( "'self'", 'data:' );
        $font_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_font_src', '' )
        );
        $font = array_merge( $font, $font_extra );

        // Connect / XHR sources.
        $connect = array( "'self'" );
        $connect_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_connect_src', '' )
        );
        $connect = array_merge( $connect, $connect_extra );

        // frame-ancestors.
        $frame = array( "'self'" );
        $frame_extra = $this->explode_sources(
            (string) $settings->get( $section, 'builder_frame_ancestors', '' )
        );
        $frame = array_merge( $frame, $frame_extra );

		$enforced = array();
		$report   = array();

        $enforced[] = "default-src 'self'";
		if ( 'enforce' === $mode_script ) {
			$enforced[] = 'script-src ' . $this->implode_unique( $script );
		} elseif ( 'report_only' === $mode_script ) {
			$report[] = 'script-src ' . $this->implode_unique( $script );
		}
		if ( 'enforce' === $mode_style ) {
			$enforced[] = 'style-src ' . $this->implode_unique( $style );
		} elseif ( 'report_only' === $mode_style ) {
			$report[] = 'style-src ' . $this->implode_unique( $style );
		}
		if ( 'enforce' === $mode_img ) {
			$enforced[] = 'img-src ' . $this->implode_unique( $img );
		} elseif ( 'report_only' === $mode_img ) {
			$report[] = 'img-src ' . $this->implode_unique( $img );
		}
		if ( 'enforce' === $mode_font ) {
			$enforced[] = 'font-src ' . $this->implode_unique( $font );
		} elseif ( 'report_only' === $mode_font ) {
			$report[] = 'font-src ' . $this->implode_unique( $font );
		}
		if ( 'enforce' === $mode_connect ) {
			$enforced[] = 'connect-src ' . $this->implode_unique( $connect );
		} elseif ( 'report_only' === $mode_connect ) {
			$report[] = 'connect-src ' . $this->implode_unique( $connect );
		}
		if ( 'enforce' === $mode_frame ) {
			$enforced[] = 'frame-ancestors ' . $this->implode_unique( $frame );
		} elseif ( 'report_only' === $mode_frame ) {
			$report[] = 'frame-ancestors ' . $this->implode_unique( $frame );
		}

        $extra = (string) $settings->get( $section, 'builder_extra_directives', '' );
        if ( '' !== $extra ) {
            $lines = preg_split( '/[\r\n]+/', $extra );
            if ( is_array( $lines ) ) {
                foreach ( $lines as $line ) {
                    $line = trim( $line );
                    if ( '' !== $line ) {
						if ( 'enforce' === $mode_extra ) {
							$enforced[] = $line;
						} elseif ( 'report_only' === $mode_extra ) {
							$report[] = $line;
						}
                    }
                }
            }
        }

		return array(
			'enforced' => implode( '; ', $enforced ),
			'report'   => implode( '; ', $report ),
		);
    }

    protected function explode_sources( $value ) {
        $sources = array();
        $value   = (string) $value;

        if ( '' === $value ) {
            return $sources;
        }

        $parts = preg_split( '/[\s\r\n]+/', $value );
        if ( ! is_array( $parts ) ) {
            return $sources;
        }

        foreach ( $parts as $part ) {
            $part = trim( $part );
            if ( '' === $part ) {
                continue;
            }

            // Basic safety: strip any trailing semicolons.
            $part = rtrim( $part, ';' );

            $sources[] = $part;
        }

        return $sources;
    }

    protected function implode_unique( array $tokens ) {
        $tokens = array_unique( array_filter( $tokens ) );
        return implode( ' ', $tokens );
    }

    protected function get_request_area( $settings, $section ) {
        $area = is_admin() ? 'admin' : 'frontend';

        if ( ! $this->is_pro_active() ) {
            return $area;
        }

        $raw_paths = (string) $settings->get( $section, 'profiles_custom_paths', '' );
        if ( '' === $raw_paths ) {
            return $area;
        }

        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        if ( '' === $request_uri ) {
            return $area;
        }

        $lines = preg_split( '/[\r\n]+/', $raw_paths );
        if ( ! is_array( $lines ) ) {
            return $area;
        }

        foreach ( $lines as $line ) {
            $path = trim( $line );
            if ( '' === $path ) {
                continue;
            }
            if ( 0 === strpos( $request_uri, $path ) ) {
                return 'custom';
            }
        }

        return $area;
    }

    protected function get_csp_report_uri() {
        if ( ! function_exists( 'home_url' ) ) {
            return '';
        }

        $url = home_url( '/' );
        $url = add_query_arg( 'aegisshield_csp_report', '1', $url );
        $url = add_query_arg( 'aegisshield_csp_nonce', wp_create_nonce( 'aegisshield_csp_report' ), $url );

        if ( function_exists( 'esc_url_raw' ) ) {
            $url = esc_url_raw( $url );
        }

        return $url;
    }

    public function maybe_handle_csp_report() {
        if ( ! $this->is_pro_active() ) {
            return;
        }

        $request_method = isset( $_SERVER['REQUEST_METHOD'] ) ? sanitize_key( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : '';
        if ( 'post' !== $request_method ) {
            return;
        }

        if ( ! isset( $_GET['aegisshield_csp_report'] ) ) {
            return;
        }

        // Nonce + capability check (admin context) to satisfy WP security best practices.
        if ( ! is_user_logged_in() || ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Forbidden.', 'aegisshield-security' ), '', array( 'response' => 403 ) );
        }
        $nonce = isset( $_GET['aegisshield_csp_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['aegisshield_csp_nonce'] ) ) : '';
        if ( ! $nonce || ! wp_verify_nonce( $nonce, 'aegisshield_csp_report' ) ) {
            wp_die( esc_html__( 'Invalid nonce.', 'aegisshield-security' ), '', array( 'response' => 403 ) );
        }

        $raw = file_get_contents( 'php://input' );
        if ( ! $raw ) {
            wp_die( '', '', array( 'response' => 204 ) );
        }

        $data = json_decode( $raw, true );
        $ctx  = array();

        if ( is_array( $data ) && isset( $data['csp-report'] ) && is_array( $data['csp-report'] ) ) {
            $report = $data['csp-report'];

            $ctx['effective_directive'] = isset( $report['effective-directive'] ) ? substr( (string) $report['effective-directive'], 0, 255 ) : '';
            $ctx['violated_directive']  = isset( $report['violated-directive'] ) ? substr( (string) $report['violated-directive'], 0, 255 ) : '';
            $ctx['blocked_uri']         = isset( $report['blocked-uri'] ) ? substr( (string) $report['blocked-uri'], 0, 255 ) : '';
            $ctx['document_uri']        = isset( $report['document-uri'] ) ? substr( (string) $report['document-uri'], 0, 255 ) : '';
            $ctx['line_number']         = isset( $report['line-number'] ) ? (int) $report['line-number'] : 0;
        } else {
            $ctx['raw'] = substr( $raw, 0, 2000 );
        }

		$this->log_violation(
			'csp_violation',
			__( 'Content-Security-Policy report received.', 'aegisshield-security' ),
			$ctx,
			'csp_violation'
		);

        wp_die( '', '', array( 'response' => 204 ) );
    }
}