<?php

namespace AegisShield\Utils;

defined( 'ABSPATH' ) || exit;

class AS_QR {

    public static function svg_base64( string $text ): string {
        try {
            $matrix = self::qr_matrix( $text );
            $svg    = self::matrix_to_svg( $matrix );
            return base64_encode( $svg );

        } catch ( \Throwable $e ) {
            // Failure fallback QR (red error box).
            $fallback = '<svg xmlns="http://www.w3.org/2000/svg" width="200" height="200">'
                      . '<rect width="100%" height="100%" fill="#8b0000"/>'
                      . '<text x="50%" y="50%" font-size="20" fill="#ffffff" text-anchor="middle" '
                      . 'dominant-baseline="middle">QR ERR</text>'
                      . '</svg>';
            return base64_encode( $fallback );
        }
    }

    protected static function qr_matrix( string $text ): array {
        $data = self::byte_mode_encode( $text );
        $ec   = self::apply_error_correction( $data );
        $raw  = self::interleave( $data, $ec );

        return self::place_into_matrix( $raw );
    }

    protected static function byte_mode_encode( string $text ): array {
        $bytes = array_values( unpack( 'C*', $text ) );
        return array_slice( $bytes, 0, 17 ); // Version 1-L limit
    }

    protected static function apply_error_correction( array $data ): array {
        return array( 0, 0, 0, 0, 0, 0, 0 );
    }

    protected static function interleave( array $data, array $ec ): array {
        return array_merge( $data, $ec );
    }

    protected static function place_into_matrix( array $raw ): array {
        $size = 21;
        $m = array_fill( 0, $size, array_fill( 0, $size, 0 ) );

        // Finder patterns in the 3 corners.
        self::add_finder( $m, 0, 0 );
        self::add_finder( $m, $size - 7, 0 );
        self::add_finder( $m, 0, $size - 7 );

        // Fill remaining with raw bits (simple row scan).
        $i = 0;
        for ( $y = 0; $y < $size; $y++ ) {
            for ( $x = 0; $x < $size; $x++ ) {
                // Skip finder pattern areas.
                if ( $m[$y][$x] !== 0 ) {
                    continue;
                }
                if ( isset( $raw[$i] ) ) {
                    $m[$y][$x] = ($raw[$i] & 1) ? 1 : 0;
                    $i++;
                }
            }
        }

        return $m;
    }

    protected static function add_finder( array &$m, int $x, int $y ): void {
        for ( $dy = 0; $dy < 7; $dy++ ) {
            for ( $dx = 0; $dx < 7; $dx++ ) {
                $is_border = ($dx === 0 || $dx === 6 || $dy === 0 || $dy === 6);
                $is_center = ($dx >= 2 && $dx <= 4 && $dy >= 2 && $dy <= 4);
                $m[$y + $dy][$x + $dx] = ($is_border || $is_center) ? 1 : 0;
            }
        }
    }

    protected static function matrix_to_svg( array $m ): string {
        $scale = 6;
        $size  = count( $m );
        $dim   = $size * $scale;

        $svg  = '<svg xmlns="http://www.w3.org/2000/svg" width="' . $dim . '" height="' . $dim . '" shape-rendering="crispEdges">';
        $svg .= '<rect width="100%" height="100%" fill="#ffffff"/>';

        foreach ( $m as $y => $row ) {
            foreach ( $row as $x => $bit ) {
                if ( $bit ) {
                    $svg .= '<rect x="' . ( $x * $scale ) . '" y="' . ( $y * $scale ) . '" '
                         . 'width="' . $scale . '" height="' . $scale . '" fill="#000000"/>';
                }
            }
        }

        $svg .= '</svg>';
        return $svg;
    }
}
